import React, { useCallback, useEffect, useState, useRef } from "react";
import { Add, Minus } from "iconsax-react";
import { useSnackbar } from "notistack";

interface Props {
  productInCart: number;
  onAddCard: VoidFunction;
  onRemoveCard: VoidFunction;
  isLoading: boolean;
  limit?: number;
  quantity?: number;
}

function AddCardButton({
  productInCart,
  onAddCard,
  onRemoveCard,
  quantity = 10,
  limit = 10,
  isLoading,
}: Props) {  
  const { enqueueSnackbar } = useSnackbar();
  const handleAddCart = useCallback(() => {
    if (productInCart >= limit) {
      enqueueSnackbar(`حداکثر میزان سفارش از این آیتم ${limit} عدد می باشد!`, {
        variant: "error",
      });
    } else if (productInCart >= quantity) {
      enqueueSnackbar(`موجودی این آیتم ${quantity} عدد می باشد!`, {
        variant: "error",
      });
    } else onAddCard();
  }, [productInCart]);
  // Ensure productInCart is always a valid number with aggressive type checking
  const safeProductInCart = (() => {
    const num = Number(productInCart);
    if (isNaN(num) || !isFinite(num)) return 0;
    return Math.max(0, Math.floor(num));
  })();
  
  const safeLimit = Math.max(1, Number(limit) || 10);
  const safeQuantity = Math.max(0, Number(quantity) || 10);
  
  // Use ref to track the last valid count to prevent stale updates
  const lastValidCount = useRef(safeProductInCart);
  const [displayCount, setDisplayCount] = useState(safeProductInCart);
  const updateTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  
  useEffect(() => {
    // Clear any pending updates
    if (updateTimeoutRef.current) {
      clearTimeout(updateTimeoutRef.current);
    }
    
    // Detect concatenation: if new value is suspiciously large (more than 10x the last value)
    // and looks like string concatenation (e.g., 2 -> 21, 3 -> 38)
    let validCount = safeProductInCart;
    
    if (lastValidCount.current > 0 && safeProductInCart > lastValidCount.current * 10) {
      // Likely a concatenation bug - extract the correct value
      // For example: 21 with last value 2 -> use 2
      // For example: 168 with last value 16 -> use 16
      const strValue = String(safeProductInCart);
      const lastStr = String(lastValidCount.current);
      
      // Check if it starts with the last value (concatenation pattern)
      if (strValue.startsWith(lastStr)) {
        // It's likely the last value + 1
        validCount = lastValidCount.current + 1;
      } else {
        // Try to extract the first digit(s)
        validCount = Math.floor(safeProductInCart / 10);
      }
    }
    
    // Only update if the value actually changed and we're not loading
    if (!isLoading && validCount !== lastValidCount.current) {
      lastValidCount.current = validCount;
      
      // Use requestAnimationFrame to ensure we update after all renders complete
      updateTimeoutRef.current = setTimeout(() => {
        setDisplayCount(validCount);
      }, 10);
    }
    
    return () => {
      if (updateTimeoutRef.current) {
        clearTimeout(updateTimeoutRef.current);
      }
    };
  }, [safeProductInCart, isLoading]);
  
  return (
    <div className="flex items-center gap-4">
      {safeQuantity === 0 ? (
        <span className="text-h5 text-secondary">ناموجود</span>
      ) : (
        <>
          {displayCount > 0 && (
            <>
              <button
                onClick={() => onRemoveCard()}
                disabled={isLoading}
                className={`flex size-12 items-center justify-center rounded-base border border-primary ${
                  isLoading ? "opacity-50 cursor-not-allowed" : ""
                }`}
              >
                <Minus fontSize={16} color="#000000" />
              </button>
              <span className={`min-w-[20px] text-center font-medium transition-all ${
                isLoading ? "blur-[2px]" : ""
              }`}>
                {displayCount}
              </span>
            </>
          )}
          <button
            onClick={handleAddCart}
            disabled={isLoading }
            className={`flex size-12 items-center justify-center rounded-base border border-primary ${
              isLoading ? "opacity-50 cursor-not-allowed" : ""
            } ${
              safeProductInCart >= safeLimit || safeProductInCart >= safeQuantity ? "bg-primary" : "bg-white"
            }`}
          >
            <Add
              size="32"
              color={
                safeProductInCart >= safeLimit || safeProductInCart >= safeQuantity
                  ? "#ffffff"
                  : "#000000"
              }
            />
          </button>
        </>
      )}
    </div>
  );
}

export default AddCardButton;
