import React, { useEffect, useRef, useState } from "react";
import { MapContainer, TileLayer, useMap, useMapEvents } from "react-leaflet";
import "leaflet/dist/leaflet.css";
import { LatLngExpression } from "leaflet";
import { defaultMapCenter } from "@/src/configs";
import { Gps } from "iconsax-react";
import Image from "next/image";
import pinImage from "@/public/assets/images/icons/marker.svg";
import AddressFormSearch from "@/src/containers/Profile/profileTabs/Address/AddressPrimary/AddressFormSearch";
import { useAreasQuery } from "@/src/hooks/Queries";

interface Props {
  onLocationChange?: (lat: string, lng: string) => void;
  ShowSearch?: boolean;
}

export default function NeshanMapContainer({
  onLocationChange,
  ShowSearch = false,
}: Props) {
  const [userLocation, setUserLocation] = useState<LatLngExpression | null>(
    null
  );
  const [mapCenter, setMapCenter] =
    useState<LatLngExpression>(defaultMapCenter);
  const [userSelectedLocation, setUserSelectedLocation] =
    useState<LatLngExpression | null>(null);
  const [defaultCity, setDefaultCity] = useState("");
  const isProgrammaticUpdate = useRef(false);
  const { data: areasOptions } = useAreasQuery();

  useEffect(() => {
    if (!defaultCity && Array.isArray(areasOptions) && areasOptions.length) {
      // Use first available area label as default city name for search suggestions
      setDefaultCity((areasOptions as any)[0]?.label || "");
    }
  }, [areasOptions, defaultCity]);

  // Add the SetMapCenter component
  function SetMapCenter({ center }: { center: LatLngExpression }) {
    const map = useMap();
    useEffect(() => {
      if (!isProgrammaticUpdate.current) {
        isProgrammaticUpdate.current = true;
        const currentZoom = map.getZoom(); // ✅ Get current zoom
        map.setView(center, currentZoom);
        setTimeout(() => {
          isProgrammaticUpdate.current = false;
        }, 500);
      }
    }, [center, map]);
    return null;
  }

  function MapEventHandler({
    onLocationChange,
  }: {
    onLocationChange?: (lat: string, lng: string) => void;
  }) {
    useMapEvents({
      moveend: (e) => {
        if (!isProgrammaticUpdate.current) {
          const center = e.target.getCenter();
          setUserSelectedLocation([center.lat, center.lng]);
          setMapCenter([center.lat, center.lng]); // ✅ Add this line!
          onLocationChange?.(center.lat.toString(), center.lng.toString());
        }
      },
    });
    return null;
  }

  // Fetch user's GPS location
  useEffect(() => {
    if ("geolocation" in navigator) {
      navigator.geolocation.getCurrentPosition(
        (position) => {
          const lat = position.coords.latitude;
          const lng = position.coords.longitude;
          setUserLocation([lat, lng]);
          // Only set initial center, don't override user selection
          if (!userSelectedLocation) {
            setMapCenter([lat, lng]);
          }
        },
        (error) => {
          console.error("Geolocation error:", error);
        }
      );
    }
  }, []);

  return (
    <div className="relative flex size-full flex-col">
      {/* ✅ Search Box */}
      {ShowSearch && (
        <div className="mb-2">
          <label className="mb-1 block text-sm font-medium">آدرس</label>
          <AddressFormSearch
            defaultCity={defaultCity}
            onSelect={(value) => setMapCenter(value)}
          />
        </div>
      )}
      <MapContainer
        center={mapCenter}
        zoom={13}
        className="size-full"
        zoomControl={false}
      >
        <TileLayer url="https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png" />
        <SetMapCenter center={mapCenter} />
        {/* Attach map event handler */}
        <MapEventHandler onLocationChange={onLocationChange} />
      </MapContainer>

      {/* Centered Marker (fixed in UI) */}
      <div className="pointer-events-none absolute left-1/2 top-1/2 z-50 h-auto w-[25px] -translate-x-1/2 -translate-y-1/2">
        <Image
          style={{
            transform: "translate3d(-50%, -50%, 0) scale(2)",
            willChange: "transform",
          }}
          src={pinImage}
          width={116}
          height={70}
          alt="marker"
          className="pointer-events-none absolute left-1/2 top-1/2 z-20"
        />
      </div>

      {/* Re-center to user's GPS location */}
      <button
        className="rounded-md shadow-md absolute bottom-4 right-4 z-40 bg-white px-5 py-3"
        onClick={() => {
          if (userLocation) {
            setMapCenter(userLocation);
          }
        }}
      >
        <div className="flex items-center">
          <p className="ml-2 text-[12px]">موقعیت من</p>
          <Gps size="20" color="#FF8A65" />
        </div>
      </button>
    </div>
  );
}
