import React from "react";
import { Controller, useFormContext } from "react-hook-form";
import { ClassNameValue } from "tailwind-merge";

import Select, {
  CSSObjectWithLabel,
  GroupBase,
  Props as SelectProps,
} from "react-select";

type CustomSelectProps<
  Option,
  IsMulti extends boolean = false,
  Group extends GroupBase<Option> = GroupBase<Option>,
> = SelectProps<Option, IsMulti, Group> & {
  name: string;
  disabled?: boolean;
  label: string;
  className?: ClassNameValue;
  fullWidth?: boolean;
};

export default function RHFSelect<
  Option,
  IsMulti extends boolean = false,
  Group extends GroupBase<Option> = GroupBase<Option>,
>(props: CustomSelectProps<Option, IsMulti, Group>) {
  const { control } = useFormContext();
  const { name, label, options, fullWidth, disabled, ...other } = props;

  // Minimal select styling (no "focus/hover state" borders) + full-width searchable input
  const styles = {
    container: (base: CSSObjectWithLabel) => ({
      ...base,
      width: "100%",
    }),
    control: (base: CSSObjectWithLabel) => ({
      ...base,
      width: "100%",
      fontSize: "12px",
      borderColor: "#0000001A",
      boxShadow: "none",
      borderRadius: 8,
      minHeight: 42,
      backgroundColor: "rgba(255, 255, 255, 0.9)",
      ":hover": {
        borderColor: "#0000001A",
      },
    }),
    valueContainer: (base: CSSObjectWithLabel) => ({
      ...base,
      flex: "1 1 auto",
    }),
    input: (base: CSSObjectWithLabel) => ({
      ...base,
      width: "100%",
      margin: 0,
      padding: 0,
    }),
    placeholder: (base: CSSObjectWithLabel) => ({
      ...base,
      color: "#9CA3AF",
    }),
    singleValue: (base: CSSObjectWithLabel) => ({
      ...base,
      color: "#111827",
    }),
    menu: (base: CSSObjectWithLabel) => ({
      ...base,
      width: "100%",
      zIndex: 50,
    }),
    option: (base: CSSObjectWithLabel, state: any) => {
      const isDisabled = state.data?.isDisabled;

      return {
        ...base,
        color: isDisabled ? "#888888" : state.isSelected ? "#0a0aa8" : "#111827",
        backgroundColor: isDisabled
          ? "#f0f0f0"
          : state.isSelected
            ? "rgba(255, 253, 116, 0.25)"
            : state.isFocused
              ? "#F3F4F6"
              : undefined,
        fontStyle: isDisabled ? "italic" : "normal",
        cursor: isDisabled ? "not-allowed" : "default",
        ":hover": {
          backgroundColor: isDisabled ? "#f0f0f0" : "#F3F4F6",
        },
        ":active": {
          backgroundColor: isDisabled ? "#f0f0f0" : "#F3F4F6",
        },
      };
    },
  };

  return (
    <Controller
      name={name}
      control={control}
      render={({ field, fieldState: { error } }) => (
        <div className={`flex ${fullWidth ? "w-full" : "max-w-md"} flex-col  `}>
          <label className=" text-h4   " htmlFor={name}>
            {label}
          </label>
          <Select
            menuPortalTarget={
              typeof window !== "undefined" ? document.body : null
            }
            placeholder=""
            menuPlacement="auto"
            menuPosition="fixed"
            isDisabled={disabled}
            name={name}
            styles={{
              ...styles,
              menuPortal: (base) => ({ ...base, zIndex: 9999 }),
            }}
            options={options}
            noOptionsMessage={() => "گزینه ای جهت نمایش وجود ندارد"}
            onChange={(selected) => field.onChange(selected)}
            value={field?.value?.value ? field.value : undefined}
            {...other}
          />
          {!!error && (
            <p className=" text-error-primary mt-4 text-small">
              {error?.message}
            </p>
          )}
        </div>
      )}
    />
  );
}
