import React from "react";
import Link from "next/link";
import { useSnackbar } from "notistack";
import { numberWithCommas } from "@/src/utils/Helpers";
import { Order } from "@/src/types/Cart";
import { BaseFood, Variety } from "@/src/types/food";
import { useCart } from "@/src/hooks";
import { useCartItemCount } from "@/src/hooks/useCartItemCount";
import LoaderSpinner from "@/src/components/LoaderSpinner";
import AddCardButton from "@/src/components/AddCardButton";

type Props = Omit<Order, "pricing" | "id"> & {
  isLoading: boolean;
  onCartChanged: VoidFunction;
};

function CartItems({ cart, isLoading, chef, onCartChanged }: Props) {
  const {
    handleAddCard,
    handleRemoveCard,
    isLoading: isLoadingCart,
  } = useCart();
  const { enqueueSnackbar } = useSnackbar();

  const AddCartHandler = async (product: BaseFood, variety: Variety) => {
    try {
      await handleAddCard(product, variety);
      onCartChanged();
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };

  const removeFromCartHandler = async (varietyID: string) => {
    try {
      await handleRemoveCard(varietyID);
      onCartChanged();
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };

  return (
    <div className="flex w-full max-w-2xl flex-col items-center rounded-half border border-black/20">
      {isLoading ? (
        <LoaderSpinner className="size-20" />
      ) : (
        <div className="flex w-full max-w-2xl flex-col items-center">
          <Link
            href={`/chef/${chef.id}`}
            dir="rtl"
            aria-label={`مشاهده صفحه ${chef.name}`}
            className="group block rounded-2 bg-white px-12 py-8 transition focus:outline-none focus:ring-2 focus:ring-primary/30"
          >
            <div className="flex items-center justify-between">
              <span className="truncate text-h3 font-bold text-gray-800 group-hover:text-primary">
                {chef.name}
              </span>
              <svg
                className="h-6 w-6 text-gray-600 transition-transform group-hover:-translate-x-1 group-hover:text-primary"
                viewBox="0 0 24 24"
                fill="none"
                stroke="currentColor"
                strokeWidth="2"
                aria-hidden="true"
              >
                <path d="M15 5l-7 7 7 7" />
              </svg>
            </div>
          </Link>
          <table className="w-full">
            <tbody>
              {cart.map((item) => (
                <tr
                  key={item.variety.varietyID}
                  className="border-b border-black/40 text-right"
                >
                  <td>
                    <div className="flex items-center justify-between p-8">
                      <div className="flex flex-col gap-4">
                        <p>{item.food.title}</p>
                        <p className="text-h5">{item.variety.title}</p>
                        <div className="flex items-center gap-2">
                          {item.variety.discount?.percent !== 0 && (
                            <span className="text-h5 font-light text-secondary line-through">
                              <span>
                                {numberWithCommas(item.variety.price)}
                              </span>
                            </span>
                          )}
                          <p className="text-h3 font-bold">
                            {numberWithCommas(
                              item.variety.discount?.discountedPrice ?? 0
                            )}{" "}
                            <span className="text-h5 font-normal">تومان</span>
                          </p>
                        </div>
                      </div>
                      <CartItemAddButton
                        food={item.food as BaseFood}
                        variety={item.variety}
                        onAddCard={() =>
                          AddCartHandler(item.food as BaseFood, item.variety)
                        }
                        onRemoveCard={() =>
                          removeFromCartHandler(item.variety.varietyID)
                        }
                        isLoading={isLoadingCart}
                      />
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}
    </div>
  );
}

// Separate component to handle cart count reactivity
function CartItemAddButton({
  variety,
  onAddCard,
  onRemoveCard,
  isLoading,
}: {
  food: BaseFood;
  variety: Variety;
  onAddCard: () => void;
  onRemoveCard: () => void;
  isLoading: boolean;
}) {
  const productInCart = useCartItemCount(variety.varietyID);

  return (
    <AddCardButton
      onAddCard={onAddCard}
      onRemoveCard={onRemoveCard}
      productInCart={productInCart}
      limit={variety.maxOrderLimit || 10}
      isLoading={isLoading}
    />
  );
}

export default CartItems;
