import Button from "@/src/components/Button";
import OrderDetailCard from "@/src/components/OrderDetailCard";
import OrderProgress from "@/src/components/OrderProgress";
import { useCart } from "@/src/hooks";
import { ServiceDetails } from "@/src/services/httpServices/CartServices/types";
import { useCartStore } from "@/src/store";
import { Pricing } from "@/src/types/Cart";

import { useSnackbar } from "notistack";
import React, { useEffect, useRef, useState } from "react";

interface Props {
  pricing?: Pricing;
  isLoading: boolean;
  onComplete: VoidFunction;
  onFinish: (description: string) => void;
  isFinish: boolean;
  adreessResult?: ServiceDetails;
  step: number;
}
function OrderDetail({
  pricing,
  isLoading,
  adreessResult,
  onComplete,
  isFinish,
  step,
  onFinish,
}: Props) {
  const { isLoading: CartLoading } = useCart();

  const [description, setDescription] = useState("");
  const { enqueueSnackbar } = useSnackbar();
  const orderTotal = Number(pricing?.orderTotal ?? 0);
  const minOrder = Number((pricing?.minOrder as number) ?? 0);
  const isBelowMin = orderTotal < minOrder;

 
  const handleComplete = () => {
    onComplete();
    if (!isFinish && step === 2) {
      enqueueSnackbar("لطفا تمامی موارد را تکمیل کنید", {
        variant: "error",
      });
    }
  };
  const delivery = useCartStore((state) => state.delivery);

  const [isLoadingManual, setIsLoadingManual] = useState(false);
  const prevTotal = useRef<number | null>(null);

  useEffect(() => {
    if (pricing?.orderTotal !== prevTotal.current) {
      setIsLoadingManual(true);
      const timeout = setTimeout(() => {
        setIsLoadingManual(false);
      }, 1000);
      prevTotal.current = pricing?.orderTotal ?? null;
      return () => clearTimeout(timeout);
    }
  }, [pricing?.orderTotal]);
  return (
    <div className="relative w-full">
      {/* Blur overlay */}
      {isLoadingManual && (
        <div className="absolute inset-0 z-10 bg-white/60 backdrop-blur-sm transition-opacity duration-200" />
      )}
      <OrderDetailCard
        adreessResult={adreessResult}
        showExtraPrice={step === 2}
        pricing={pricing}
        isLoading={isLoading}
        footer={
          <div className="flex w-full flex-col items-center justify-center gap-8">
            {delivery && step === 2 && (
              <textarea
                value={description}
                onChange={(e) => setDescription(e.target.value)}
                placeholder={`${delivery.descriptionText} ${delivery.descriptionRequired ? "*" : ""}`}
                rows={8}
                autoComplete="off"
                className="block w-full resize-none  rounded-base border border-black/40
                      px-8 py-4 text-h5   focus:border-gray-800 "
              />
            )}

            {isBelowMin && isFinish && (
              <OrderProgress
                total={orderTotal}
                minOrder={pricing?.minOrder as number}
              />
            )}
            {!isBelowMin && isFinish && (
              <Button
                onClick={() => onFinish(description)}
                fullWidth
                disabled={delivery?.descriptionRequired && !description}
                intent="primary"
                className="py-8"
                isLoading={CartLoading}
              >
                ثبت سفارش
              </Button>
            )}
            {!isFinish && (
              <Button
                onClick={() => handleComplete()}
                fullWidth
                intent="primary"
                className="py-8"
                isLoading={CartLoading}
              >
                ثبت سفارش
              </Button>
            )}
          </div>
        }
      />
    </div>
  );
}

export default OrderDetail;
