import { useAuthStore } from "@/src/store";
import UseCartStore from "@/src/store/UseCartStore";
import {
  Map1,
  ShoppingCart,
  UserSquare,
  Location,
  ArrowLeft2,
} from "iconsax-react";
import Link from "next/link";
import React, { useEffect, useState } from "react";
import { motion } from "framer-motion";
import { useAuth } from "@/src/hooks";
import { Modal } from "@/src/components/Modal";
import {
  useAddressQuery,
  useAreasQuery,
  useOrderAddressQuery,
} from "@/src/hooks/Queries";
import LoaderSpinner from "@/src/components/LoaderSpinner";
import NextImg from "@/src/components/NextImage";
import emptyImage from "@/public/assets/images/default/no-address.svg";
import { useSettingsStore } from "@/src/providers/SettingsProvider";
import { useRouter } from "next/navigation";
import { useQueryClient } from "@tanstack/react-query";
import AddressForm from "@/src/containers/Profile/profileTabs/Address/AddressForm";
import { Select } from "@/src/components/RHF";
import { FormProvider, useForm } from "react-hook-form";
import { useSelectionStore } from "@/src/store/useSelectionStore";

function UserActions() {
  const { cardItems } = UseCartStore();
  const { authUser } = useAuthStore();
  const [key, setKey] = useState(0);
  const [callbackUrl, setCallbackUrl] = useState<string>("/");

  const {
    getTokenFromCookies,
    changeSelectedAddress,
    isLoading: isLoadingSelectedAddress,
    getUser,
  } = useAuth();
  const { refetch } = useOrderAddressQuery();
  const { mode } = useSettingsStore((state) => state.settings);
  const router = useRouter();
  const queryClient = useQueryClient();

  const token = getTokenFromCookies();
  const [defaultCity, setDefaultCity] = useState("");
  const [search, setSearch] = useState("");
  const [debouncedValue, setDebouncedValue] = useState("");

  const [showAddAddress, setShowAddAddressModal] = useState(false);
  const [showSelectAddress, setshowSelectAddressModal] = useState(false);
  const [showDrawer, setShowDrawer] = useState(false);
  const { data: areasOptions } = useAreasQuery();
  const ensureRestaurant = useSelectionStore((s) => s.ensureRestaurant);
  const updateSelection = useSelectionStore((s) => s.updateSelection);
  const [restaurantId, setRestaurantId] = useState<string | null>(null);

  const normalizedParams = {
    address_title: debouncedValue?.trim() ? debouncedValue : undefined,
    address_area: defaultCity,
  };

  const {
    data,
    isLoading,
    refetch: refetchListAddress,
  } = useAddressQuery(normalizedParams);

  useEffect(() => {
    // preferred: use window in client. Guard for SSR.
    if (typeof window === "undefined") return;

    // full path including search/query and hash if you want:
    const path =
      window.location.pathname + window.location.search + window.location.hash;

    // fallback to "/" if somehow empty
    const safePath = path || "/";

    setCallbackUrl(encodeURIComponent(safePath));
  }, []);
  const authHref = (target: string) =>
    authUser.mobile ? target : `/auth?callback=${callbackUrl}`;

  useEffect(() => {
    setKey((prevKey) => prevKey + 1);
  }, [cardItems]);
  useEffect(() => {
    if (restaurantId) {
      ensureRestaurant(restaurantId);
    }
  }, [restaurantId, ensureRestaurant]);

  const addAddress = () => {
    if (token) {
      return setshowSelectAddressModal(true);
    }

    setShowAddAddressModal(true); // Show the modal only if no unsaved address
  };
  useEffect(() => {
    // Check if unsavedAddress exists in localStorage
    if (token) {
      return setShowDrawer(true);
    }
    const unsavedAddress = localStorage.getItem("unsavedAddress");
    setTimeout(() => {
      if (unsavedAddress) {
        setShowDrawer(true);
      }
    }, 1000);
  }, [token]);

  useEffect(() => {
    // robustly read pathname (works in client)
    const pathname =
      typeof window !== "undefined" ? window.location.pathname : "";
    const match = pathname.match(/\/chef\/(\d+)(?:\/|$)/i);
    setRestaurantId(match ? match[1] : null);
  }, []);

  // debounce manually

  useEffect(() => {
    const handler = setTimeout(() => {
      // always push the current value (even empty) after debounce
      setDebouncedValue(search.trim());
    }, 500);

    return () => clearTimeout(handler);
  }, [search]);

  useEffect(() => {
    if (debouncedValue === "") {
      refetchListAddress(); // 👈 always hits the API with empty query
    }
  }, [debouncedValue, refetchListAddress]);

  const defaultAddressTitle = React.useMemo(() => {
    if (!data || !authUser.defaultAddress) return null;

    const defaultAddress = data.find(
      (address) => Number(address.id) === Number(authUser.defaultAddress)
    );
    return defaultAddress ? defaultAddress.address : null;
  }, [data, authUser.defaultAddress]);

  function handleRadioClick(id: string): void {
    changeSelectedAddress({ id: id });

    if (restaurantId) {
      // updateSelection expects Partial<Omit<Selection, "restaurantId" | "updatedAt">>
      updateSelection(restaurantId, { selectedAddressId: id });
    }

    setTimeout(() => {
      router.refresh();
      queryClient.invalidateQueries();
    }, 1000);
  }
  const methods = useForm();

  return (
    <div className="flex items-center gap-7 xl:gap-16 ">
      {showDrawer && mode !== "single" ? (
        <div
          onClick={addAddress}
          className="flex w-full cursor-pointer flex-col items-center justify-center truncate rounded-half p-1 text-h3 leading-6 lg:-ml-7 lg:border-black/10 lg:px-8 lg:py-5"
        >
          <div className="flex items-center rounded-half border p-1 text-h3 leading-6 lg:border-black/10 lg:px-8 lg:py-5">
            <Map1 className="me-2 h-8 w-8 text-h3 lg:text-h3" color="#000000" />
            {defaultAddressTitle ? (
              <p className="max-w-[90px] truncate text-h6 text-black xl:max-w-[200px]">
                {defaultAddressTitle}
              </p>
            ) : (
              <span className="mr-2 hidden w-full overflow-hidden text-ellipsis lg:block">
                تغییر آدرس
              </span>
            )}
          </div>
        </div>
      ) : (
        ""
      )}
      <Link
        className="flex w-full max-w-[30px]  items-center justify-center gap-2 truncate rounded-half border text-h3 leading-6 lg:max-w-80 lg:border-black/10 lg:px-8 lg:py-5"
        href={authHref("/profile")}
      >
        <UserSquare className="text-h3 lg:text-h1 " color="#000000" />
        <span className="hidden w-full overflow-hidden text-ellipsis lg:block">
          {authUser.mobile
            ? `${authUser?.name ?? ""}  ${authUser?.family ?? ""} `
            : "ورود / ثبت نام"}
        </span>
      </Link>

      <Link
        key={key}
        href={authHref("/cart")}
        className="relative "
      >
        <motion.div
          animate={{
            x: cardItems.length > 0 ? [-2, 2, -2, 2, -2, 2, 0] : 0,
          }}
          transition={{ duration: 1 }}
        >
          <ShoppingCart
            className="text-h3 lg:text-h1 "
            color="#000000"
            variant="Broken"
          />
        </motion.div>
        {!!cardItems.length && (
          <span
            className=" absolute -bottom-1 -start-1 inline-flex size-8  items-center justify-center rounded-base
            bg-primary text-h5 font-bold text-white "
          >
            {cardItems.length ?? 0}
          </span>
        )}
      </Link>

      <Modal
        title="انتخاب آدرس"
        onClose={() => setshowSelectAddressModal(false)}
        open={showSelectAddress}
        wrapperClassName="m-10"
      >
        <div className="relative max-h-[400px] overflow-y-auto lg:w-[600px]">
          {isLoadingSelectedAddress && (
            <div className="absolute inset-0 z-10 flex items-center justify-center bg-white/60 backdrop-blur-sm">
              <LoaderSpinner />
            </div>
          )}

          <div className="mb-2 grid grid-cols-12 items-end gap-4">
            <FormProvider {...methods}>
              <div className="col-span-4">
                <Select
                  placeholder="استان"
                  label="استان"
                  options={areasOptions}
                  onChange={(selected) => {
                    if (!selected) return;
                    setDefaultCity(selected.value || "");
                  }}
                  name="area.parent"
                  fullWidth
                />
              </div>
              <div className="col-span-8">
                <label className="mb-1 block text-[12px] font-medium">
                  آدرس
                </label>
                <input
                  type="text"
                  value={search}
                  onChange={(e) => setSearch(e.target.value)}
                  placeholder="جستجوی آدرس..."
                  className="text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500 w-full rounded-[5px] border border-gray-400/25 bg-white px-3 py-3 placeholder:text-small focus:outline-none focus:ring-1"
                />
              </div>
            </FormProvider>
          </div>

          <div
            className={`flex  flex-col gap-2 px-4  ${
              isLoadingSelectedAddress ? "pointer-events-none blur-sm" : ""
            }`}
          >
            {isLoading && <LoaderSpinner />}
            {data?.map((item) => (
              <div
                key={item.id}
                onClick={() => handleRadioClick(item.id)}
                className={`flex cursor-pointer flex-col gap-6 border-b border-black/20 p-10`}
              >
                <div className="flex items-center">
                  <label className="flex w-full cursor-pointer items-center gap-4">
                    <input
                      type="radio"
                      name="address"
                      id={`address-${item.id}`}
                      readOnly
                      onClick={(e) => {
                        e.stopPropagation();
                        handleRadioClick(item.id); // manually trigger
                      }}
                      checked={item.id == authUser.defaultAddress}
                      className="h-8 w-8 cursor-pointer accent-primary"
                    />
                    <p className="text-h3 font-bold text-black">{item.title}</p>
                  </label>
                </div>
                <p className="flex gap-2 text-h4 text-black/90">
                  <Location size={16} className="text-primary" />
                  {item.address}
                </p>
              </div>
            ))}

            {!isLoading && !data?.length && (
              <div className="flex flex-col items-center justify-center gap-12">
                <NextImg
                  src={emptyImage}
                  alt="no-address"
                  width={160}
                  height={160}
                />
                <p className="text-h3">تاکنون آدرسی برای شما ثبت نشده است.</p>
                <div className=" flex justify-end pb-15">
                  <button
                    onClick={() => setShowAddAddressModal(true)}
                    className="flex  items-center gap-4 text-h5 text-text-link "
                  >
                    افزودن آدرس جدید
                    <ArrowLeft2 size="16" color="#007AFF" variant="Broken" />
                  </button>
                </div>
              </div>
            )}
          </div>
        </div>
        {!isLoading && data?.length ? (
          <div className=" qwerrr flex justify-end py-5">
            <button
              onClick={() => setShowAddAddressModal(true)}
              className="flex  items-center gap-4 text-h5 text-text-link "
            >
              افزودن آدرس جدید
              <ArrowLeft2 size="16" color="#007AFF" variant="Broken" />
            </button>
          </div>
        ) : (
          ""
        )}
      </Modal>
      <Modal
        title="افزودن آدرس"
        onClose={() => setShowAddAddressModal(false)}
        open={showAddAddress}
      >
        <AddressForm
          onAddAddress={() => {
            refetchListAddress();
            refetch();
            setShowAddAddressModal(false);
          }}
        />
      </Modal>
    </div>
  );
}

export default UserActions;
