import { orderData } from "@/src/types/general";
import React, { useEffect, useState } from "react";

import NextImg from "@/src/components/NextImage";
import emptyImage from "@/public/assets/images/default/no-order.svg";
import { useTranslation } from "react-i18next";
import ChefOrder from "../ProductOrder";
import ChefLoader from "../ProductsLoader";
import { BaseFood } from "@/src/types/food";
import MiniProductCard from "@/src/components/MiniProductCard";
import ProductCategory from "../ProductCategory";
import { useCart } from "@/src/hooks";
import { useSettingsStore } from "@/src/providers/SettingsProvider";
import { useRouter, useSearchParams } from "next/navigation";
import { useSnackbar } from "notistack";
import { Variety } from "@/src/types/food";

interface Props {
  chefs: BaseFood[] | [];
  onChangeOrder: (type: orderData) => void;
  defaultCategory: string
  onChangeCategories: (category: string) => void;
  order: orderData;
  isLoading: boolean;
  hasLocationPermission: boolean;
}
function ChefsItems({
  chefs,
  onChangeOrder,
  order,
  defaultCategory,
  onChangeCategories,
  isLoading,
  hasLocationPermission,
}: Props) {
  const { t } = useTranslation();

  const [selectedID, setSelectedID] = useState<string | undefined>(undefined);

  const { emptyProductImage } = useSettingsStore((state) => state.settings);

  const router = useRouter();
  const searchParams = useSearchParams();

  useEffect(() => {
    if (selectedID) {
      onChangeCategories(selectedID);
    }

  }, [selectedID])
  const { handleAddCard, handleRemoveCard } = useCart();
  const { enqueueSnackbar } = useSnackbar();

  const AddCartHandler = async (product: BaseFood, variety: Variety) => {
    try {
      await handleAddCard(product, variety);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };

  const removeFromCartHandler = async (varietyID: string) => {
    try {
      await handleRemoveCard(varietyID);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };

  const removeParams = () => {
    const params = new URLSearchParams(searchParams.toString());

    // Remove 'category' and 'orderBy'
    params.delete("category");
    params.delete("orderBy");

    // Update the URL without refreshing the page
    router.push(`?${params.toString()}`);

  }

  return (
    <div className="flex min-h-[600px] w-full ">
      <div className="flex w-full gap-8">
        <div className="hidden w-full max-w-xs lg:block">
          <ChefOrder
            onChangeOrder={onChangeOrder}
            order={order}
            hasLocationPermission={hasLocationPermission}
          />
          <ProductCategory
            setSelectedID={setSelectedID}
            selectedID={selectedID}
          />
        </div>

        <div className="grid h-fit w-full grid-cols-1 gap-x-8 gap-y-16 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 ">
          {isLoading && <ChefLoader />}
          {chefs.map((item) => (

            <MiniProductCard
              onSelect={() =>
                router.push(
                  `/product/${item.id}/${item.title.replaceAll(" ", "-").replaceAll("%", "درصد")}`
                )
              }
              className="w-full "
              titleClassName=" lg:max-w-[100px] xl:max-w-[120px] 2xl:max-w-[220px]"
              onAddCard={(variety) => AddCartHandler(item, variety)}
              onRemoveCard={(variety) => removeFromCartHandler(variety)}
              key={item.id}
              image={item.image}
              title={item.title}
              description={item.description}
              available={true}
              varieties={item.varieties.slice(0, 1)}
            />


          ))}{" "}
          {!isLoading && !chefs.length && (
            <div className="col-span-3 flex w-full flex-col items-center justify-center gap-12 py-16 ">
              <NextImg
                src={emptyProductImage}
                alt="no-address"
                width={115}
                height={140}
                imgStyles={{ objectFit: "contain" }}
              />
              <p className="text-h3 ">{t("product_notFound")}</p>
              <button onClick={removeParams}
                type="button"

                className="inline-flex gap-4 border border-primary p-4 rounded-half text-h4 text-secondary"
              >
                مشاهده همه محصولات
              </button>
            </div>
          )}
        </div>
      </div>
    </div>
  );
}

export default ChefsItems;
