import React, { useMemo, useState } from "react";
import { AnimatePresence, motion } from "framer-motion";
import NextImg from "@/src/components/NextImage";
import { useSliderQuery } from "@/src/hooks/Queries";
import { useMediaQuery } from "@uidotdev/usehooks";
import { ArrowLeft2, ArrowRight2 } from "iconsax-react";
import Image from "next/image";
import sliderShape from "@/public/assets/images/icons/slider-shape.svg";
import AddCardButton from "@/src/components/AddCardButton";
import { numberWithCommas } from "@/src/utils/Helpers";
import { useCart } from "@/src/hooks";
import { useCartStore } from "@/src/store";
import Link from "next/link";
import { SliderType } from "@/src/services/httpServices/SettingsServices/types";
import SliderItem from "./SliderItem";

interface Props {
  type: SliderType;
}

function ChefSlider({ type }: Props) {
  const [rotation, setRotation] = useState(0);
  const [activeIndex, setActiveIndex] = useState(1);
  const { data, isLoading } = useSliderQuery(type);
  const sliderData = useMemo(() => data?.slice(0, 4) || [], [data]);

  const { handleAddCard, handleRemoveCard } = useCart();
  const { itemCount } = useCartStore();
  const [isVisible, setIsVisible] = useState(true);

  const handleClick = () => {
    setIsVisible(false);

    // Set a timeout to make the button reappear after 1 second
    setTimeout(() => {
      setIsVisible(true);
    }, 500);
  };
  const enterAnimation = () => {
    setRotation(rotation + 90);

    handleClick();
    setActiveIndex((prevState) =>
      prevState + 1 < sliderData.length ? prevState + 1 : 0
    );
  };
  const enterAnimationMin = () => {
    setRotation(rotation - 90);
    setActiveIndex((prevState) =>
      prevState - 1 >= 0 ? prevState - 1 : sliderData.length - 1
    );
    handleClick();
  };
  const isDesktop = useMediaQuery("only screen and (min-width: 1024px)");
  return (
    <div className="h-[450px] w-full  bg-primary bg-searchMobile  bg-cover bg-center bg-no-repeat px-2 lg:bg-search   ">
      {!isLoading && !!sliderData.length && (
        <div className="flex size-full flex-col items-center justify-between overflow-hidden px-3 lg:container lg:flex-row">
          <div className="flex min-h-[100px] w-full flex-col  gap-3  py-2 text-white lg:w-1/3 lg:py-0  ">
            <AnimatePresence>
              {isVisible && (
                <motion.div
                  exit={{ opacity: 0 }}
                  key="desddc"
                  transition={{ duration: 1 }}
                  className="flex flex-col gap-3 text-white  "
                >
                  <motion.h2
                    initial={{ x: 300, opacity: 0 }}
                    animate={{ x: 0, opacity: 1 }}
                    transition={{ duration: 0.75 }}
                    className="text-hero font-bold lg:text-large"
                  >
                    {sliderData[activeIndex].title}
                  </motion.h2>
                  <motion.p
                    initial={{ x: 300, opacity: 0 }}
                    animate={{ x: 0, opacity: 1 }}
                    transition={{ duration: 1 }}
                    className="hidden text-h2 lg:block "
                  >
                    {sliderData[activeIndex].Summary}
                  </motion.p>
                  <motion.p
                    initial={{ x: 300, opacity: 0 }}
                    animate={{ x: 0, opacity: 1 }}
                    transition={{ duration: 1 }}
                    className="hidden text-body lg:block"
                  >
                    رستوران: {sliderData[activeIndex].Chef.title}
                  </motion.p>

                  <motion.div
                    initial={{ y: 50, opacity: 0 }}
                    animate={{ y: 0, opacity: 1 }}
                    transition={{ duration: 1.2 }}
                    className="flex w-full items-center justify-between "
                  >
                    <div className="flex items-center justify-center gap-1">
                      {!!sliderData[activeIndex].discount.percent && (
                        <div className="flex size-auto items-center justify-center rounded-px bg-secondary p-1  text-small font-bold text-white  lg:text-h4">
                          {sliderData[activeIndex].discount.percent} %
                        </div>
                      )}

                      <div>
                        {!!sliderData[activeIndex].discount.percent && (
                          <s className="text-h5 font-light ">
                            {numberWithCommas(sliderData[activeIndex].price)}
                          </s>
                        )}

                        <p className=" text-h4 font-bold lg:text-h3">
                          {numberWithCommas(
                            sliderData[activeIndex].discount.discountedPrice
                          )}
                          تومان
                        </p>
                      </div>
                    </div>
                    <div className="flex items-center gap-2">
                      <Link
                        className="hidden text-h4 font-bold lg:block"
                        href={`/product/${sliderData[activeIndex].id}`}
                      >
                        جدول ارزش غذایی
                      </Link>
                      <AddCardButton
                        buttonIntent="primary"
                        productInCart={itemCount(
                          sliderData[activeIndex].varietyID
                        )}
                        limit={sliderData[activeIndex].maxOrderLimit}
                        onAddCard={() => handleAddCard(sliderData[activeIndex])}
                        onRemoveCard={() =>
                          handleRemoveCard(sliderData[activeIndex].varietyID)
                        }
                      />
                    </div>
                  </motion.div>
                </motion.div>
              )}
            </AnimatePresence>
          </div>

          <div className="flex size-full items-center justify-center  gap-2 lg:w-1/3  lg:flex-col lg:items-end  ">
            <div className="hidden w-full items-center  justify-end gap-6 px-8  lg:flex">
              <button
                className=" flex items-center justify-center p-1 "
                onClick={enterAnimationMin}
              >
                <ArrowRight2 size="24" color="#ffffff" variant="Broken" />
              </button>
              <button
                className=" flex items-center justify-center p-1 "
                onClick={enterAnimation}
              >
                <ArrowLeft2 size="24" color="#ffffff" variant="Broken" />
              </button>
            </div>

            <button
              className=" flex items-center justify-center lg:hidden "
              onClick={enterAnimationMin}
            >
              <ArrowRight2 size="24" color="#ffffff" variant="Broken" />
            </button>
            <AnimatePresence>
              {isVisible && (
                <motion.div
                  key="shape"
                  initial={{ scale: 0.5 }}
                  animate={{ scale: 1 }}
                  exit={{ scale: 0.5, opacity: 0 }}
                  transition={{ duration: 1, bounce: 0.5, type: "spring" }}
                  className=" relative z-20 flex  size-[200px] items-center justify-center  rounded-full  lg:size-[350px]"
                >
                  <Image
                    src={sliderShape}
                    width={116}
                    height={70}
                    alt="a"
                    className="absolute -right-4 bottom-1 z-20"
                  />{" "}
                  <Image
                    src={sliderShape}
                    width={116}
                    height={70}
                    alt="a"
                    className="absolute -left-4 top-1 z-20 rotate-180"
                  />
                  <motion.div
                    animate={{ rotate: 360 }}
                    transition={{
                      ease: "linear",
                      duration: 20,
                      repeat: Infinity,
                      delay: 0,
                    }}
                    className="flex size-full  items-center justify-center rounded-full"
                  >
                    <NextImg
                      alt={sliderData[activeIndex].title}
                      src={sliderData[activeIndex].image}
                      width={isDesktop ? 350 : 150}
                      height={isDesktop ? 350 : 150}
                      imgWrapperClassName="rounded-full overflow-hidden"
                      imgClassName="w-full h-full"
                    />
                  </motion.div>
                </motion.div>
              )}
            </AnimatePresence>
            <button
              className="  flex items-center justify-center lg:hidden"
              onClick={enterAnimation}
            >
              <ArrowLeft2 size="24" color="#ffffff" variant="Broken" />
            </button>
          </div>
          {isDesktop && (
            <div className=" relative flex h-full  w-1/3 items-center justify-end overflow-hidden    ">
              <div className="relative  flex h-full  items-center  overflow-hidden      ">
                <div className=" absolute -top-[15%] left-[20%] z-10 flex size-[600px] rotate-180 items-center justify-center rounded-full   bg-searchCircle ">
                  <span className="size-[80%] rounded-full  bg-secondary"></span>
                </div>
                {sliderData && sliderData.length && (
                  <motion.div
                    animate={{ rotate: rotation }}
                    transition={{
                      ease: "linear",
                      duration: 1,
                    }}
                    className=" relative left-[170px] z-20 size-[450px] rounded-full xl:left-[145px] 2xl:left-[100px] "
                  >
                    {/* left */}
                    <div className="absolute -left-[100px] top-1/2 flex size-[200px] -translate-y-1/2 items-center justify-center rounded-full ">
                      <SliderItem
                        image={sliderData[0].image}
                        title={sliderData[0].title}
                      />
                    </div>
                    {/* right */}
                    <div className="absolute -right-[100px] top-1/2 flex size-[200px] -translate-y-1/2 items-center justify-center rounded-full ">
                      <SliderItem
                        image={sliderData[1].image}
                        title={sliderData[1].title}
                      />{" "}
                    </div>
                    {/* bottom */}
                    <div className="absolute -bottom-[100px] right-1/2 flex size-[200px] translate-x-1/2 items-center justify-center rounded-full ">
                      <SliderItem
                        image={sliderData[2].image}
                        title={sliderData[2].title}
                      />{" "}
                    </div>
                    {/* top */}
                    <div className="absolute -top-[100px] right-1/2 flex size-[200px] translate-x-1/2 items-center justify-center rounded-full ">
                      <SliderItem
                        image={sliderData[3].image}
                        title={sliderData[3].title}
                      />{" "}
                    </div>
                  </motion.div>
                )}
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
}

export { ChefSlider };
