import { FormProvider } from "@/src/components/RHF";
import React, { useCallback, useEffect, useState } from "react";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";

import { getAddressFromLocation } from "@/src/app/actions";
import { useAddress, useAuth } from "@/src/hooks";
import { useSnackbar } from "notistack";
import { useAddressByIdQuery } from "@/src/hooks/Queries";
import { AddressForm as AddressFormType } from "@/src/types/Address";
import { useSettingsStore } from "@/src/providers/SettingsProvider";
import { addressSchema } from "./schema";
import AddressFormDetail from "./AddressFormDetail";
import AddressFormMap from "./AddressFormMap";
import Cookies from "js-cookie";

const defaultValues: AddressFormType = {
  title: "",
  lat: "",
  lng: "",
  phone: "",
  address: "",
  area: {
    parent: {
      label: "",
      value: "",
      lat: "",
      lng: "",
    },
    child: null,
  },
};
interface Props {
  onAddAddress: VoidFunction;
  addressId?: string;
}
function AddressForm({ onAddAddress, addressId }: Props) {
  const { forceMap } = useSettingsStore((state) => state.settings);
  const [step, setStep] = useState<"map" | "form">(addressId ? "form" : "map");
  const { data: addressData, isLoading } = useAddressByIdQuery(addressId);

  const methods = useForm<AddressFormType>({
    defaultValues,
    resolver: yupResolver(addressSchema),
    mode: "onChange",
    disabled: isLoading,
  });
  const { handleSubmit, setValue, reset } = methods;

  const { createAddress, updateAddress } = useAddress();
  const { enqueueSnackbar } = useSnackbar();

  const { getTokenFromCookies } = useAuth();


  const onSubmit = handleSubmit(async (data) => {
    const token = getTokenFromCookies();

    const saveToDatabase = async (dataToSave: AddressFormType) => {
      try {
        if (addressId) {
          await updateAddress(addressId, {
            address: dataToSave.address,
            lat: dataToSave.lat ?? "",
            lng: dataToSave.lng ?? "",
            title: dataToSave.title,
            ...(!forceMap && {
              area_id: dataToSave.area.child
                ? dataToSave.area.child.value
                : dataToSave.area.parent.value,
              lat: dataToSave.area.child
                ? dataToSave.area.child.lat
                : dataToSave.area.parent.lat,
              lng: dataToSave.area.child
                ? dataToSave.area.child.lng
                : dataToSave.area.parent.lng,
            }),
            phone_number: dataToSave.phone,
          });
        } else {
          await createAddress({
            phone_number: dataToSave.phone,
            address: dataToSave.address,
            lat: dataToSave.lat ?? "",
            lng: dataToSave.lng ?? "",
            title: dataToSave.title,
            ...(!forceMap && {
              area_id: dataToSave.area.child
                ? dataToSave.area.child.value
                : dataToSave.area.parent.value,
              lat: dataToSave.area.child
                ? dataToSave.area.child.lat
                : dataToSave.area.parent.lat,
              lng: dataToSave.area.child
                ? dataToSave.area.child.lng
                : dataToSave.area.parent.lng,
            }),
          });
        }

        enqueueSnackbar("تغییرات با موفقیت انجام شد", {
          variant: "success",

        });
        reset();
        onAddAddress();
      } catch (error) {
        console.log(error);
        enqueueSnackbar("خطا در ثبت اطلاعات", {
          variant: "error",
        });
      }
    };


    if (token) {
      // User is logged in; save directly to the database
      await saveToDatabase(data);
    } else {
      // User is not logged in; save data to localStorage
      console.log("User not logged in, saving to localStorage");
      localStorage.setItem("unsavedAddress", JSON.stringify(data));
      Cookies.set("lat", data.lat as string, { path: "/" });
      Cookies.set("lng", data.lng as string, { path: "/" });

      enqueueSnackbar("اطلاعات ذخیره شد و پس از ورود ثبت خواهد شد", {
        variant: "info",
      });

      setTimeout(() => {
        window.location.reload();
      }, 1000);

      reset();
      onAddAddress();
    }

  });
  useEffect(() => {
    reset({
      title: addressData?.title,
      lat: addressData?.lat,
      lng: addressData?.lng,
      address: addressData?.address,
      phone: addressData?.phone ?? "",
      ...(forceMap
        ? {
          area: {
            parent: {
              label: addressData?.lat,
              value: addressData?.lat,
              lat: addressData?.lat,
              lng: addressData?.lng,
            },
          },
        }
        : {
          area: {
            parent: addressData?.area?.parent,
            child: addressData?.area?.child,
          },
        }),
    });
  }, [addressData]);
  const handleAddress = async (lat: string, lng: string) => {
    const addr = await getAddressFromLocation(lat, lng);
    setValue("address", addr.formattedAddress);
    setValue("lat", lat);
    setValue("lng", lng);
    if (forceMap) {
      setValue("area", {
        parent: {
          label: lat,
          value: lat,
          lat,
          lng,
        },
        child: {
          label: lat,
          value: lat,
          lat,
          lng,
        },
      });
    }
  };
  const RenderStep = useCallback(() => {
    let content = <></>;
    switch (step) {
      case "form":
        content = <AddressFormDetail onShowMap={() => setStep("map")} />;
        break;
      case "map":
        content = (
          <AddressFormMap
            onFinish={() => setStep("form")}
            onClick={handleAddress}
          />
        );
        break;

      default:
        break;
    }
    return content;
  }, [step]);
  return (
    <>
      <div className="flex flex-col">
        <FormProvider methods={methods} onSubmit={onSubmit}>
          {forceMap ? (
            <RenderStep />
          ) : (
            <AddressFormDetail onShowMap={() => setStep("map")} />
          )}
        </FormProvider>
      </div>
    </>
  );
}

export default AddressForm;
