import React, { useEffect, useState } from "react";
import { CSSObjectWithLabel } from "react-select";
import AsyncSelect from "react-select/async";
import { getLocationFromAddressFromLocation } from "@/src/app/actions";
import { useDebouncedCallback } from "use-debounce";
import { LatLngExpression } from "leaflet";

const styles = {
  container: (base: CSSObjectWithLabel) => ({
    ...base,
    width: "100%",
  }),
  control: (base: CSSObjectWithLabel) => ({
    ...base,
    width: "100%",
    fontSize: "14px",
    borderColor: "#0000001A",
    boxShadow: "none",
    borderRadius: 8,
    backgroundColor: "rgba(255, 255, 255, 0.9)",
    ":hover": {
      borderColor: "#0000001A",
    },
  }),
  valueContainer: (base: CSSObjectWithLabel) => ({
    ...base,
    flex: "1 1 auto",
  }),
  input: (base: CSSObjectWithLabel) => ({
    ...base,
    width: "100%",
    margin: 0,
    padding: 0,
  }),
  menu: (base: CSSObjectWithLabel) => ({
    ...base,
    zIndex: 500,
  }),
  option: (base: CSSObjectWithLabel, state: any) => ({
    ...base,
    backgroundColor: state.isFocused ? "#F3F4F6" : "transparent",
    color: "#111827",
    ":hover": {
      backgroundColor: "#F3F4F6",
    },
  }),
};

interface Props {
  onSelect: (value: LatLngExpression) => void;
  defaultCity?: string;
}

function AddressFormSearch({ onSelect, defaultCity }: Props) {
  const [value, setValue] = useState<any>(
    defaultCity ? { label: defaultCity, value: defaultCity } : null
  );
  const [inputValue, setInputValue] = useState(value?.label || ""); // 👈 default typing text

  useEffect(() => {
    if (defaultCity) {
      setValue({ label: defaultCity, value: defaultCity });
      setInputValue(defaultCity);
    } else {
      setValue(null);
    }
  }, [defaultCity]);

  const promiseOptions = useDebouncedCallback(async (inputValue: string) => {
    const query = defaultCity ? `${defaultCity} ${inputValue}` : inputValue;

    const options = await getLocationFromAddressFromLocation(query);
    return options;
  }, 500);

  return (
    <AsyncSelect
      cacheOptions
      isClearable
      inputValue={inputValue}
      onInputChange={(newValue) => setInputValue(newValue)} // ✅ allow typing
      onChange={(item) => {
        setValue(item);
        onSelect({
          lat: item?.location.y ?? 0,
          lng: item?.location.x ?? 0,
        });
      }}
      defaultOptions
      loadingMessage={() => "درحال جستجو..."}
      noOptionsMessage={() => "آدرسی یافت نشد!"}
      styles={styles}
      loadOptions={promiseOptions}
      placeholder="جستجوی آدرس..."
      name="address"
    />
  );
}

export default AddressFormSearch;
