import React, { useEffect, useState } from "react";
import { Modal } from "@/src/components/Modal";
import { useAddressQuery, useAreasQuery } from "@/src/hooks/Queries";
import LoaderSpinner from "@/src/components/LoaderSpinner";
import NextImg from "@/src/components/NextImage";
import emptyImage from "@/public/assets/images/default/no-address.svg";
import { useAddress } from "@/src/hooks";
import { useSnackbar } from "notistack";
import AddressCard from "./AddressCard";
import AddressForm from "./AddressForm";
import { useSearchParams } from "next/navigation";
import { FormProvider, useForm } from "react-hook-form";
import { Select } from "@/src/components/RHF";

function Address() {
  const [showAddressModal, setShwAddressModal] = useState({
    create: false,
    edit: false,
  });
  const [addressId, setAddressId] = useState("");
  const { deleteAddress } = useAddress();
  const { enqueueSnackbar } = useSnackbar();
  const searchParams = useSearchParams();
  const { data: areasOptions } = useAreasQuery();
  const [defaultCity, setDefaultCity] = useState("");
  const [search, setSearch] = useState("");
  const [debouncedValue, setDebouncedValue] = useState("");

  useEffect(() => {
    const tab = searchParams.get("profileTab");
    if (tab === "address") {
      refetch();
    }
  }, [searchParams]);

  const handleDelete = async (id: string) => {
    try {
      await deleteAddress(id);
      refetch();
      enqueueSnackbar("آدرس با موفقیت حذف شد", {
        variant: "success",
      });
    } catch (error) {
      console.log(error);
      enqueueSnackbar("خطا در حذف آدرس", {
        variant: "error",
      });
    }
  };
  const methods = useForm();
  const normalizedParams = {
    address_title: debouncedValue?.trim() ? debouncedValue : undefined,
    address_area: defaultCity,
  };

  const { data, isLoading, refetch } = useAddressQuery(normalizedParams);

  useEffect(() => {
    const handler = setTimeout(() => {
      // always push the current value (even empty) after debounce
      setDebouncedValue(search.trim());
    }, 500);

    return () => clearTimeout(handler);
  }, [search]);

  useEffect(() => {
    if (debouncedValue === "") {
      refetch(); // 👈 always hits the API with empty query
    }
  }, [debouncedValue, refetch]);

  return (
    <div className="flex flex-col ">
      <div className="flex w-full items-center justify-between border-b border-black/20 pb-6">
        <p className="text-h3  ">آدرس</p>

        <button
          onClick={() =>
            setShwAddressModal((prevState) => ({
              ...prevState,
              create: true,
            }))
          }
          className="hover:text-bold flex items-center justify-center whitespace-nowrap rounded-half border border-primary px-12 py-4 text-h3  text-primary transition-all  hover:bg-error-extraLight"
        >
          آدرس جدید
        </button>
      </div>
      <div className="mt-7 grid grid-cols-12 items-end gap-4">
        <FormProvider {...methods}>
          <div className="col-span-4">
            <Select
              placeholder="استان"
              label="استان"
              options={areasOptions}
              onChange={(selected) => {
                if (!selected) return;
                setDefaultCity(selected.value || "");
              }}
              name="area.parent"
              fullWidth
            />
          </div>
          <div className="col-span-8">
            <label className="mb-1 block text-[12px] font-medium">آدرس</label>
            <input
              type="text"
              value={search}
              onChange={(e) => setSearch(e.target.value)}
              placeholder="جستجوی آدرس..."
              className="text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500 w-full rounded-[5px] border border-gray-400/25 bg-white px-3 py-3 placeholder:text-small focus:outline-none focus:ring-1"
            />
          </div>
        </FormProvider>
      </div>
      <div className="flex flex-col gap-2 px-4 py-3">
        {isLoading && <LoaderSpinner />}
        {data?.map((item) => (
          <AddressCard
            lat={item.lat}
            lng={item.lng}
            plaque={item.plaque}
            postalCode={item.postalCode}
            phone={item.phone}
            unit={item.unit}
            key={item.id}
            address={item.address}
            title={item.title}
            onEdit={() => {
              setAddressId(item.id);
              setShwAddressModal((prevState) => ({
                ...prevState,
                edit: true,
              }));
            }}
            onDelete={() => handleDelete(item.id)}
          />
        ))}
        {!isLoading && !data?.length && (
          <div className="flex flex-col items-center justify-center gap-12">
            <NextImg
              src={emptyImage}
              alt="no-address"
              width={160}
              height={160}
            />
            <p className="text-h3">تاکنون آدرسی برای شما ثبت نشده است.</p>
          </div>
        )}
      </div>

      <Modal
        title="افزودن آدرس"
        onClose={() =>
          setShwAddressModal((prevState) => ({
            ...prevState,
            create: false,
          }))
        }
        open={showAddressModal.create}
      >
        <AddressForm
          onAddAddress={() => {
            refetch();
            setShwAddressModal((prevState) => ({
              ...prevState,
              create: false,
            }));
          }}
        />
      </Modal>
      <Modal
        title="ویرایش آدرس"
        onClose={() =>
          setShwAddressModal((prevState) => ({
            ...prevState,
            edit: false,
          }))
        }
        open={showAddressModal.edit}
      >
        <AddressForm
          addressId={addressId}
          onAddAddress={() => {
            refetch();
            setShwAddressModal((prevState) => ({
              ...prevState,
              edit: false,
            }));
          }}
        />
      </Modal>
    </div>
  );
}

export default Address;
