"use client";

import Header from "@/src/containers/Layouts/Header";
import { ChefDetail } from "@/src/types/Chef";
import React, { useMemo } from "react";
import { Swiper, SwiperSlide } from "swiper/react";
import { Autoplay } from "swiper/modules";
import Link from "next/link";
import Image from "next/image";
import { useSliderQuery } from "@/src/hooks/Queries";
import { ApiSlider } from "@/src/services/httpServices/SettingsServices/types";

interface Props {
  chefData: ChefDetail;
  available: boolean;
}

function parseCovers(cover?: string): string[] {
  if (!cover) return [];
  const raw = String(cover).trim();
  if (!raw) return [];

  // Try JSON array first
  try {
    const parsed = JSON.parse(raw);
    if (Array.isArray(parsed)) {
      return parsed
        .map((x) => String(x || "").trim())
        .filter((x) => x && x !== "null" && x !== "undefined");
    }
  } catch {
    // ignore
  }

  // Fallback: split by commas
  return raw
    .split(",")
    .map((x) => x.trim())
    .filter((x) => x && x !== "null" && x !== "undefined");
}

function normalizeSlides(home?: ApiSlider[], chefCover?: string) {
  if (home && home.length > 0) {
    return home
      .map((s) => ({
        src: s.image,
        title: s.title,
        link: s.link,
      }))
      .filter((x) => x.src);
  }

  const covers = parseCovers(chefCover);
  return covers.map((src) => ({
    src,
    title: "",
    link: "",
  }));
}

function SingleRestaurantBanner({ chefData }: Props) {
  const { data: homeSlides, isLoading } = useSliderQuery("home");

  const slides = useMemo(
    () => normalizeSlides(homeSlides, chefData.cover),
    [homeSlides, chefData.cover]
  );

  if (isLoading && (!slides || slides.length === 0)) {
    return (
      <div className="w-full bg-white">
        <Header />
        <div className="container px-4 pb-6 lg:px-0">
          <div className="h-[150px] w-full rounded-2 bg-gray-100 lg:h-[380px]" />
        </div>
      </div>
    );
  }

  const safeSlides =
    slides && slides.length > 0
      ? slides
      : [
          {
            src: "",
            title: chefData.title || "banner",
            link: "",
          },
        ];

  const loop = safeSlides.length > 1;

  return (
    <div className="w-full bg-white">
      <Header />
      <div className="container px-4 pb-6 pt-4 lg:px-0">
        <div className="relative h-[150px] w-full overflow-hidden rounded-2 bg-gray-100 lg:h-[380px]">
          <Swiper
            loop={loop}
            modules={[Autoplay]}
            autoplay={loop ? { delay: 3500, disableOnInteraction: false } : false}
            className="size-full"
          >
            {safeSlides.map((item, idx) => (
              <SwiperSlide
                key={`${item.src || "fallback"}-${idx}`}
                className="relative size-full"
              >
                {item.src ? (
                  item.link ? (
                    <Link href={item.link} className="relative block size-full">
                      <Image
                        src={item.src}
                        alt={item.title || chefData.title || "banner"}
                        fill
                        sizes="100vw"
                        className="object-cover"
                        priority={idx === 0}
                      />
                    </Link>
                  ) : (
                    <Image
                      src={item.src}
                      alt={item.title || chefData.title || "banner"}
                      fill
                      sizes="100vw"
                      className="object-cover"
                      priority={idx === 0}
                    />
                  )
                ) : (
                  <div className="flex size-full items-center justify-center text-h4 text-gray-400">
                    {chefData.title}
                  </div>
                )}
              </SwiperSlide>
            ))}
          </Swiper>
        </div>
      </div>
    </div>
  );
}

export default SingleRestaurantBanner;
