import RestaurantAccordion from "@/src/components/RestaurantAccordion";
import { useCartQuery } from "@/src/hooks/Queries";
import {
  DeliveryDetails,
  DeliveryMethod,
} from "@/src/hooks/Queries/useDeliveryMethodByTime";
import { ServiceDetails } from "@/src/services/httpServices/CartServices/types";
import { useAuthStore, useCartStore } from "@/src/store";
import { ChefDetail } from "@/src/types/Chef";
import {
  getWithExpiryLocalStorage,
  numberWithCommas,
} from "@/src/utils/Helpers";
import Link from "next/link";
import React, { FC, useEffect, useState } from "react";
import closeImage from "@/public/assets/images/icons/closed.svg";
import Image from "next/image";
import { useRestaurantSelection } from "@/src/containers/useRestaurantSelection";

type Props = {
  chefData: ChefDetail;
  setAvailableModal: React.Dispatch<React.SetStateAction<boolean>>;
  areaAddressData: ServiceDetails | undefined;
  deliveryMethod: DeliveryDetails | undefined;
  deliveryOptions: DeliveryMethod[] | undefined;
};
const translateDayToPersian = (day: string): string => {
  const days: Record<string, string> = {
    Saturday: "شنبه",
    Sunday: "یک‌شنبه",
    Monday: "دوشنبه",
    Tuesday: "سه‌شنبه",
    Wednesday: "چهارشنبه",
    Thursday: "پنج‌شنبه",
    Friday: "جمعه",
  };

  return days[day] || day;
};

const SingleRestaurantOrder: FC<Props> = ({
  chefData,
  areaAddressData,
  setAvailableModal,
  deliveryMethod,
  deliveryOptions,
}) => {
  const [isLoading, setIsLoading] = useState(true);
  const { cardItems } = useCartStore();
  const {
    setDeliveryTime,
    deliveryTime,
    selectedAddressId,
    selectedDeliveryId,
  } = useRestaurantSelection(chefData.id);

  const {
    queryData: { data, refetch },
    refetchWithParamsWithItem,
    isManualFetching,
  } = useCartQuery();

  useEffect(() => {
    // Only trigger the effect if cardItems is not empty
    if (cardItems.length > 0 && data?.chef.id) {
      setIsLoading(true);
      
      // Refetch with params if available, otherwise just refetch
      if (selectedAddressId && selectedDeliveryId) {
        refetchWithParamsWithItem({
          address_id: selectedAddressId,
          branch_id: data.chef.id,
          delivery_type_id: selectedDeliveryId,
        });
      } else {
        // Refetch without params when address/delivery not selected yet
        refetch();
      }

      const timeout = setTimeout(() => {
        setIsLoading(false);
      }, 1000);

      return () => clearTimeout(timeout);
    } else {
      setIsLoading(false);
    }
  }, [cardItems.length, selectedAddressId, selectedDeliveryId, data?.chef.id]);

  useEffect(() => {
    if (areaAddressData?.data && !areaAddressData.active) {
      const newDay = areaAddressData.data.day;
      const newTime = areaAddressData.data.service_time_from;

      // Only call setDeliveryTime when something actually changed
      if (deliveryTime?.day !== newDay || deliveryTime?.time !== newTime) {
        setDeliveryTime({ day: newDay, time: newTime });
      }
    }
  }, [areaAddressData]);

  const deliveryType = deliveryOptions?.find(
    (item) => Number(item.id) === Number(selectedDeliveryId)
  );

  return (
    <div className="w-full bg-white py-6 md:sticky md:top-20 md:z-20">
      <div className="mb-10 hidden max-w-sm lg:block ">
        {areaAddressData?.open_branch ? (
          <RestaurantAccordion
            chefId={chefData.id}
            areaAddressData={areaAddressData}
            setAvailableModal={setAvailableModal}
            deliveryType={deliveryType}
          />
        ) : (
          <div className="mt-8 flex flex-col items-center">
            <Image
              src={closeImage}
              width={116}
              height={70}
              alt="marker"
              className="pointer-events-none"
            />
            <p>رستوران بسته است</p>
          </div>
        )}
      </div>
      {cardItems.length > 0 ? (
        <div
          className={`relative flex h-max w-full max-w-sm flex-col items-center rounded-half border border-black/20 p-12 transition-all duration-300 ${
            isManualFetching ? "pointer-events-none select-none blur-sm" : ""
          }`}
        >
          <>
            <div className="flex w-full items-center justify-between border-b  p-8">
              <p className="text-h3 ">مجموع : </p>
              <p className="text-h4">
                {numberWithCommas(data?.pricing?.total ?? 0)} تومان
              </p>
            </div>{" "}
            <div className="flex w-full items-center justify-between  p-8">
              <p className="text-h3">مالیات : </p>
              <p className="text-h4">
                {numberWithCommas(data?.pricing?.tax ?? 0)} تومان
              </p>
            </div>
            {data?.pricing?.cashBack !== 0 && (
              <div className="flex w-full items-center justify-between  p-8">
                <p className="text-h3">کش بک : </p>
                <p className="text-h4">
                  {numberWithCommas(data?.pricing?.cashBack ?? 0)}- تومان
                </p>
              </div>
            )}
            <>
              <div className="flex w-full items-center justify-between p-8">
                <p className="text-h3">هزینه ارسال : </p>
                <p className="text-h4">
                  {data?.pricing?.sendPrice
                    ? `${numberWithCommas(data?.pricing?.sendPrice ?? 0)} تومان`
                    : "رایگان"}
                </p>
              </div>{" "}
            </>
            <div className="flex w-full items-center justify-between border-b border-black/20 p-8">
              <p className="text-h3">هزینه بسته بندی : </p>
              <p className="text-h4">
                {data?.pricing?.PackPrice
                  ? `${numberWithCommas(data?.pricing?.PackPrice ?? 0)} تومان`
                  : "رایگان"}
              </p>
            </div>
            {!!data?.pricing?.discountedPrice && (
              <div className="flex w-full items-center justify-between border-b border-black/20 p-8">
                <p className="text-h3">کد تخفیف : </p>
                <p className="text-h4">
                  {numberWithCommas(data?.pricing?.discountedPrice ?? 0)} تومان
                </p>
              </div>
            )}
            <div className="flex w-full items-center justify-between border-b border-black/20 p-8">
              <p className="text-h3">قابل پرداخت : </p>
              <p className="text-h4">
                {numberWithCommas(data?.pricing?.orderTotal ?? 0)} تومان
              </p>
            </div>
            <Link
              className="hidden lg:flex relative w-full items-center gap-4 text-h5 lg:text-body"
              href="/cart"
            >
              <div className="flex w-full flex-col items-center justify-center gap-8">
                <button className="flex h-max w-full items-center justify-center rounded-half bg-primary px-12 py-8 leading-6 text-white disabled:bg-gray-800 lg:px-16">
                  ثبت سفارش
                </button>
              </div>
            </Link>
          </>
        </div>
      ) : (
        ""
      )}
    </div>
  );
};

export default SingleRestaurantOrder;
