// app/stores/useSelectionStore.ts
"use client";

import { create } from "zustand";
import { persist, createJSONStorage } from "zustand/middleware";

type DeliveryTime = { day: string; time: string; currentTime?: boolean }; // optional here

type Selection = {
  restaurantId: string;
  selectedAddressId?: string;
  deliveryId?: string;
  deliveryTime?: DeliveryTime; // <-- object, not ISO
  updatedAt: number;
};

type State = {
  selections: Record<string, Selection>;
  ensureRestaurant: (restaurantId: string) => void; // write just id on open
  updateSelection: (
    restaurantId: string,
    patch: Partial<Omit<Selection, "restaurantId" | "updatedAt">>
  ) => void; // commit on submit
  getSelection: (restaurantId: string) => Selection | undefined;
  clearSelection: (restaurantId: string) => void;
  getSelectionState: () => Record<string, Selection>; // <-- Add this line
};

export const useSelectionStore = create<State>()(
  persist(
    (set, get) => ({
      selections: {},

      ensureRestaurant: (restaurantId) => {
        const { selections } = get();
        if (!selections[restaurantId]) {
          set({
            selections: {
              ...selections,
              [restaurantId]: { restaurantId, updatedAt: Date.now() },
            },
          });
        }
      },

      updateSelection: (restaurantId, patch) => {
        const prev = get().selections[restaurantId] ?? {
          restaurantId,
          updatedAt: 0,
        };

        // If patch contains deliveryTime, ensure currentTime defaults to false
        const patched = {
          ...prev,
          ...patch,
          restaurantId,
          updatedAt: Date.now(),
        };

        if (patch.deliveryTime) {
          patched.deliveryTime = {
            ...patch.deliveryTime,
            currentTime: patch.deliveryTime.currentTime ?? false,
          };
        }

        set({
          selections: {
            ...get().selections,
            [restaurantId]: patched,
          },
        });
      },

      getSelection: (restaurantId) => get().selections[restaurantId],

      clearSelection: (restaurantId) => {
        const { [restaurantId]: _, ...rest } = get().selections;
        set({ selections: rest });
      },
      // Add this method
      getSelectionState: () => get().selections,
    }),
    {
      name: "restaurant-selections.v1",
      storage: createJSONStorage(() => localStorage),
    }
  )
);
