import React, {
  TextareaHTMLAttributes,
  useCallback,
  useMemo,
  useState,
} from "react";
import { Controller, useFormContext } from "react-hook-form";
import { ClassNameValue, twMerge } from "tailwind-merge";
import { useDebouncedCallback } from "use-debounce";

type Props = TextareaHTMLAttributes<HTMLTextAreaElement> & {
  name: string;
  validator?: (value: string) => Promise<{ isValid: boolean }>;
  validatorError?: string;
  disabled?: boolean;
  label: string;
  className?: ClassNameValue;
  fullWidth?: boolean;
};

interface ValidationStatus {
  error: string;
  hasError: boolean;
}

export default function RHFTextArea({
  name,

  validator,
  validatorError = "Validation failed",
  disabled = false,
  label,
  fullWidth,
  className,
  ...other
}: Props) {
  const { control } = useFormContext();

  const defaultApiValidationStatus: ValidationStatus = useMemo(
    () => ({
      error: validatorError,
      hasError: false,
    }),
    [validatorError]
  );

  const [isValidating, setIsValidating] = useState<boolean>(false);
  const [validationStatus, setValidationStatus] = useState<ValidationStatus>(
    defaultApiValidationStatus
  );

  const clearErrors = () =>
    setValidationStatus((prevState) => ({
      ...prevState,
      hasError: false,
    }));

  const triggerErrors = () =>
    setValidationStatus((prevState) => ({
      ...prevState,
      hasError: true,
    }));

  const handleValidator = useDebouncedCallback(
    useCallback(async (value: string) => {
      if (!validator || !value) {
        setIsValidating(false);
        clearErrors();
        return;
      }

      try {
        setIsValidating(true);
        clearErrors();

        const checkResults = await validator(value);
        if (!checkResults.isValid) {
          triggerErrors();
        } else {
          clearErrors();
        }
      } catch (error) {
        console.log(error);
      } finally {
        setIsValidating(false);
      }
      // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []),
    500
  );
  const inputClassnames = twMerge(
    `block resize-none ${
      fullWidth ? "w-full" : "w-fit"
    } hover:border-blue-500  py-1 px-2 text-h5  border
            border-black/20 rounded-[2px]   focus:border-gray-800 `,
    className
  );
  return (
    <Controller
      name={name}
      control={control}
      render={({ field, fieldState: { error, invalid } }) => (
        <div className="flex flex-col gap-1 ">
          <label className=" text-h4  " htmlFor={name}>
            {label}
          </label>

          <textarea
            {...field}
            id={name}
            autoComplete="off"
            value={field.value}
            onChange={(event) => {
              field.onChange(event.target.value);

              if (validator && !invalid) {
                handleValidator(event.target.value);
              }
            }}
            disabled={disabled}
            {...other}
            className={inputClassnames}
          />
          {isValidating && <span>loading</span>}
          {!!error && (
            <p className=" text-error-primary mt-1 text-small">
              {error?.message ??
                (validationStatus.hasError && validationStatus.error)}
            </p>
          )}
        </div>
      )}
    />
  );
}
