"use client";

import { Logo } from "@/src/components/Logo";
import { TextField, FormProvider } from "@/src/components/RHF";
import React, { useEffect, useMemo } from "react";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as Yup from "yup";
import { useAuthStore } from "@/src/store";
import { useAuth, useCountDown } from "@/src/hooks";
import { useUrlStore } from "@/src/hooks/Extras";
import { useRouter } from "next/navigation";
import dayjs from "dayjs";
import Button from "@/src/components/Button";
import { Edit } from "iconsax-react";

interface FormFields {
  mobile: string;
  code: string;
}

function VerifyCode() {
  const { mobile } = useAuthStore((state) => state.authUser);
  const { currentCount, restartCountDown, hasCountDownFinished } = useCountDown(
    {
      EndDate: dayjs().add(3, "minute").toDate(),
    }
  );
  const { checkActivationCode, sendActivationCode, isLoading } = useAuth();
  const { reset, getData } = useUrlStore();
  const router = useRouter();
  useEffect(() => {
    if (!mobile) {
      reset();
    }
  }, [mobile]);
  const registerSchema = useMemo(
    () =>
      Yup.object().shape({
        mobile: Yup.string().required(
          "لطفا شماره موبایل خود را بصورت درست وارد کنید "
        ),
        code: Yup.string().required("ورود کد اجباری است "),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );
  const defaultValues: FormFields = useMemo(
    () => ({
      mobile: mobile ?? "",
      code: "",
    }),
    [mobile]
  );
  const methods = useForm<FormFields>({
    defaultValues,
    resolver: yupResolver(registerSchema),
  });

  const {
    handleSubmit,
    formState: { isValid },
    setError,
  } = methods;

  const onSubmit = handleSubmit(async (data) => {
    try {
      const { family, name } = await checkActivationCode(
        data.mobile,
        data.code
      );

      const callback = getData("callback");
      if (name && family) {
        if (callback) {
          router.replace(callback);
        } else router.refresh();
      } else {
        router.replace(
          `/auth/complete${callback ? `?callback=${callback}` : ""}`
        );
      }
    } catch (error) {
      setError("code", {
        type: "manual",
        message: error.data.message,
      });
    }
  });
  const handleResend = async () => {
    await sendActivationCode(mobile ?? "");
    restartCountDown();
  };
  return (
    <div className="flex w-full flex-col gap-16">
      <div className="flex flex-col items-end justify-end gap-16  ">
        <Logo width={75} height={13} />
      </div>

      <FormProvider methods={methods} onSubmit={onSubmit}>
        <div className="flex flex-col  gap-8 ">
          <p className="w-full  text-right text-h2  font-bold">تایید شماره</p>
          <div>
            <TextField
              onlyNumber
              fullWidth
              type="text"
              maxLength={4}
              className="text-center tracking-[14px]"
              name="code"
              inputMode="numeric"
              label={`کد تایید به شماره ${mobile} ارسال شد`}
            />
            <div className="mt-4 flex items-center justify-between ">
              <button
                type="button"
                onClick={() => reset()}
                className="inline-flex gap-4 text-h4  text-info"
              >
                <Edit size="16" color="#007AFF" />
                اصلاح شماره
              </button>
              {hasCountDownFinished ? (
                <button
                  type="button"
                  onClick={handleResend}
                  className="inline-flex gap-4 text-h4  text-secondary"
                >
                  ارسال دوباره
                </button>
              ) : (
                <p className=" text-left text-h4 text-info">
                  {currentCount.minutes}:{currentCount.seconds} شکیبا باشید
                </p>
              )}
            </div>
          </div>

          <Button
            type="submit"
            isLoading={isLoading}
            disabled={!isValid || isLoading}
          >
            ادامه
          </Button>
        </div>
      </FormProvider>
    </div>
  );
}

export default VerifyCode;
