// components/DeliveryOptionsList.tsx
import { DeliveryMethod } from "@/src/hooks/Queries/useDeliveryMethodByTime";
import { Address } from "@/src/types/Address";
import { DeliveryType } from "@/src/types/Cart";
import { Edit2 } from "iconsax-react";
import { useEffect, useRef, useState } from "react";

interface Props {
  deliveryData: DeliveryMethod[];
  selectedDeliveryId: number | null;
  address: Address | undefined;
  handleRadioClick: (e: React.MouseEvent, option: DeliveryMethod) => void;
  setShowAddressList: (value: boolean) => void;
  hideAddressSelectSection?:boolean
}

const DeliveryOptionsList = ({
  deliveryData,
  selectedDeliveryId,
  address,
  handleRadioClick,
  hideAddressSelectSection,
  setShowAddressList,
}: Props) => {
  const [isLoading, setIsLoading] = useState(false);
  const prevSelectedId = useRef<number | null>(null);

  useEffect(() => {
    // Trigger only if selectedDeliveryId changed
    if (
      selectedDeliveryId !== null &&
      selectedDeliveryId !== prevSelectedId.current
    ) {
      setIsLoading(true);
      const timeout = setTimeout(() => {
        setIsLoading(false);
      }, 1000);
      prevSelectedId.current = selectedDeliveryId;

      return () => clearTimeout(timeout); // cleanup
    }
  }, [selectedDeliveryId]);

  return (
    <div className="relative">
      {/* Blur & interaction block layer */}
      {isLoading && (
        <div className="absolute inset-0 z-10 cursor-wait bg-white/50 backdrop-blur-sm" />
      )}

      <div
        className={`rounded-md grid grid-cols-2 overflow-hidden transition-all ${
          isLoading ? "pointer-events-none select-none" : ""
        }`}
      >
        {deliveryData.map((option, index) => {
          const isLeftCol = index % 2 === 0;
          const isOdd = deliveryData.length % 2 !== 0;
          const isLast = index === deliveryData.length - 1;
          const isSecondLast = index === deliveryData.length - 2;

          const shouldHaveBottomBorder = isOdd
            ? !isLast
            : !(isLast || isSecondLast);

          const isSelected = Number(selectedDeliveryId) === Number(option.id);

          return (
            <div
              key={option.id}
              onClick={(e) => handleRadioClick(e, option)}
              className={`text-sm relative flex cursor-pointer h-30 items-start p-4 transition hover:bg-gray-50
                ${shouldHaveBottomBorder ? "border-b" : ""}
                ${isLeftCol ? "border-l" : ""}
                ${isSelected ? "bg-gray-50" : ""}
              `}
            >
              <div
                className={`ml-6 mr-2  h-8 w-8 min-w-8 rounded-full border-2 ${
                  isSelected ? "border-main bg-main" : "border-gray-300"
                }`}
              />
              <div className="flex flex-col space-y-1">
                <span className="font-bold text-h5 truncate max-w-[100px] sm:max-w-80">{option.title}</span>
              </div>

              <div className={`absolute right-5 bottom-1 ${hideAddressSelectSection ? 'hidden' : ''}`}>
                {address?.address ? (
                  <div
                    className={`mt-4 flex items-center gap-1 xl:flex-nowrap ${
                      isSelected && option.is_address ? "block" : "hidden"
                    }`}
                  >
                    <p className="text-h6 text-text-secondary lg:max-w-[120px] xl:truncate">
                      {address.address}
                    </p>
                    <button
                      onClick={(e) => {
                        e.stopPropagation();
                        setShowAddressList(true);
                      }}
                    >
                      <Edit2 className="text-primary" size={18} />
                    </button>
                  </div>
                ) : (
                  <span
                    onClick={(e) => {
                      e.stopPropagation();
                      setShowAddressList(true);
                    }}
                    className={`${
                      option.is_address && isSelected ? "inline-block" : "hidden"
                    } cursor-pointer rounded-1 bg-primary px-8 py-2 text-h6 text-white shadow-2 transition duration-150 hover:bg-[#b8001c]`}
                  >
                    انتخاب آدرس
                  </span>
                )}
              </div>
            </div>
          );
        })}
      </div>
    </div>
  );
};

// ✅ Prevent re-render unless props change
export default DeliveryOptionsList;
