import { Modal } from "@/src/components/Modal";
import { FormOption } from "@/src/types/general";
import { yupResolver } from "@hookform/resolvers/yup";
import { t } from "i18next";
import { ArrowLeft2 } from "iconsax-react";
import { enqueueSnackbar } from "notistack";
import React, { FC, useEffect, useMemo, useState } from "react";
import { SubmitHandler, useForm } from "react-hook-form";
import * as Yup from "yup";
import { FormProvider } from "@/src/components/RHF";
import AddressForm from "@/src/containers/Profile/profileTabs/Address/AddressForm";
import { Location } from "iconsax-react";
import UseOrderAddressByParamQuery from "@/src/hooks/Queries/UseOrderAddressByParamQuery";
import { useAuthStore, useCartStore } from "@/src/store";
import { Address } from "@/src/types/Address";
import { DeliveryDetails } from "@/src/hooks/Queries/useDeliveryMethodByTime";
import { useCart } from "@/src/hooks";
import { useRestaurantSelection } from "@/src/containers/useRestaurantSelection";

interface FormFields {
  address: FormOption;
}

interface AddressParams {
  delivery_type_id: string;
  address_id?: string;
  branch_id: string;
}

const defaultValues: FormFields = {
  address: {
    label: "",
    value: "",
  },
};

type Prop = {
  onSetAddress: (params: AddressParams) => void;
  selectedDeliveryId: number | null;
  chefId: string;
  showAddressList: boolean;
  setShowAddressList: React.Dispatch<React.SetStateAction<boolean>>;
  address: Address | undefined;
  deliveryData: DeliveryDetails | undefined;
};

const OrderAddressModal: FC<Prop> = ({
  onSetAddress,
  deliveryData,
  selectedDeliveryId,
  chefId,
  showAddressList,
  setShowAddressList,
  address,
}) => {
  const [showAddAddress, setShowAddAddressModal] = useState(false);
  const { authUser } = useAuthStore();
  const {
    setSelectedAddressId: storeSelectedAddressId,
    setSelectedDeliveryId,
  } = useRestaurantSelection(chefId);

  const {
    data: addressOrder,
    isLoading: isLoadingAddressOrder,
    refetchWithParamsOrderAreaAddress,
  } = UseOrderAddressByParamQuery();

  const storedAddressOrder = useCartStore((state) => state.addressOrder);
  const { setDelivery: setDeliveryCart } = useCart();

  useEffect(() => {
    const hasAddress = deliveryData?.delivery_methods?.find(
      (item) => Number(item.id) === Number(selectedDeliveryId)
    );

    if (
      hasAddress?.is_address &&
      !isLoadingAddressOrder &&
      (!addressOrder || addressOrder.length === 0)
    ) {
      refetchWithParamsOrderAreaAddress({
        branch_id: Number(chefId),
        user_id: Number(authUser.id),
        delivery_type_id: Number(selectedDeliveryId),
      });
    }
  }, [
    addressOrder,
    isLoadingAddressOrder,
    chefId,
    authUser.id,
    selectedDeliveryId,
  ]);

  const addressSchema = useMemo(
    () =>
      Yup.object().shape({
        address: Yup.object().shape({
          label: Yup.string().required("لطفا ادرس خود را وارد نمایید "),
          value: Yup.string()
            .required("لطفا ادرس خود را وارد نمایید ")
            .matches(/^[0-9]*$/, "لطفا ادرس خود را بصورت درست وارد کنید "),
        }),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );

  const methods = useForm<FormFields>({
    defaultValues,
    resolver: yupResolver(addressSchema),
    mode: "onChange",
  });

  const {
    handleSubmit,
    formState: { isValid, isSubmitting },
    reset,
  } = methods;

  const onSubmit: SubmitHandler<FormFields> = async (values) => {
    try {
      const addressId = String(values.address?.value ?? "");

      onSetAddress({
        delivery_type_id: String(selectedDeliveryId),
        address_id: addressId,
        branch_id: chefId,
      });

      setTimeout(() => {
        setDeliveryCart({
          delivery_type_id: String(selectedDeliveryId),
          address_id: addressId || null,
        });
      }, 1000);

      // ✅ persisted per-restaurant via Zustand
      storeSelectedAddressId(addressId);
      if (selectedDeliveryId) setSelectedDeliveryId(String(selectedDeliveryId));

      enqueueSnackbar("تغییرات با موفقیت انجام شد", { variant: "success" });
      reset();
    } catch (error: any) {
      enqueueSnackbar(error?.message || "خطا در ثبت اطلاعات", {
        variant: "error",
      });
    }
  };

  const sortedAddressOrder = addressOrder
    ? [...addressOrder].sort((a, b) => Number(b.isActive) - Number(a.isActive))
    : [];

  return (
    <div>
      <Modal
        title="انتخاب آدرس"
        open={showAddressList}
        onClose={() => setShowAddressList(false)}
      >
        <FormProvider methods={methods} onSubmit={handleSubmit(onSubmit)}>
          <div className="max-h-[43vh] w-[250px] overflow-y-auto lg:w-[400px]">
            {isLoadingAddressOrder ? (
              <div className="p-4 text-center">در حال بارگذاری...</div>
            ) : addressOrder && addressOrder.length > 0 ? (
              <div className="flex w-full flex-col gap-8">
                <div className="grid grid-cols-1">
                  {sortedAddressOrder.map((addressItem) => {
                    const isSelected = addressItem.value === address?.id;
                    const isDisabled = !addressItem.isActive;

                    return (
                      <div
                        key={addressItem.value}
                        onClick={
                          isDisabled
                            ? undefined
                            : async () => {
                                try {
                                  methods.setValue("address", addressItem); // Set selected address in form
                                  storeSelectedAddressId(addressItem.value);
                                  await handleSubmit(onSubmit)(); // Submit form
                                  setShowAddressList(false); // Close address list
                                } catch (error) {
                                  console.error(
                                    "Form submission error:",
                                    error
                                  );
                                }
                              }
                        }
                        className={`flex flex-col gap-6 border-b p-6 transition ${
                          isDisabled
                            ? "cursor-not-allowed bg-gray-50 opacity-50"
                            : "cursor-pointer hover:bg-gray-50"
                        } ${isSelected ? "bg-gray-50" : ""}`}
                      >
                        <div className="flex items-center gap-4">
                          <div
                            className={`flex h-8 w-8 items-center justify-center rounded-full border ${
                              isSelected ? "accent-primary" : "border-gray-300"
                            }`}
                          >
                            {isSelected && !isDisabled && (
                              <div className="h-4 w-4 rounded-full bg-primary" />
                            )}
                          </div>
                          <p className="text-h3 font-bold text-black">
                            {addressItem.label}
                          </p>

                          {isDisabled && (
                            <div className="ms-auto w-fit border border-gray-300 bg-gray-100 px-6 py-1 text-h6 font-medium ">
                              خارج از محدوده
                            </div>
                          )}
                        </div>
                        <p className="flex gap-2 text-h6 text-black/90">
                          <Location size={16} className="text-primary" />
                          {addressItem.description}
                        </p>
                      </div>
                    );
                  })}
                </div>

                {storedAddressOrder && !storedAddressOrder.length && (
                  <span className="my-4 text-small font-light text-secondary">
                    شما آدرسی در محدوده {t("restaurant")} ندارید !
                  </span>
                )}
              </div>
            ) : (
              <div className="p-4 text-center">آدرسی یافت نشد</div>
            )}
          </div>
        </FormProvider>
        <div className="flex justify-end pt-6">
          <button
            onClick={() => setShowAddAddressModal(true)}
            className="flex  items-center gap-4 text-h5 text-text-link "
          >
            افزودن آدرس جدید
            <ArrowLeft2 size="16" color="#007AFF" variant="Broken" />
          </button>
        </div>
      </Modal>

      <Modal
        title="افزودن آدرس"
        onClose={() => setShowAddAddressModal(false)}
        open={showAddAddress}
      >
        <AddressForm
          onAddAddress={async () => {
            await refetchWithParamsOrderAreaAddress({
              branch_id: Number(chefId),
              user_id: Number(authUser.id),
              delivery_type_id: Number(selectedDeliveryId),
            });

            setShowAddAddressModal(false);
          }}
        />
      </Modal>
    </div>
  );
};

export default OrderAddressModal;
