import { ArrowRight2, TickSquare } from "iconsax-react";
import React, { useEffect, useRef, useState } from "react";
import { Address } from "@/src/types/Address";
import { useDeliveryTypesQuery, useGatewayQuery } from "@/src/hooks/Queries";
import LoaderSpinner from "@/src/components/LoaderSpinner";
import { DeliveryType } from "@/src/types/Cart";
import { useTranslation } from "react-i18next";
import DiscountForm from "../DiscountForm";
import { useAuthStore, useCartStore } from "@/src/store";
import { ServiceDetails } from "@/src/services/httpServices/CartServices/types";
import UseOrderAddressByParamQuery from "@/src/hooks/Queries/UseOrderAddressByParamQuery";
import DeliveryOptionsList from "./components/DeliveryOptionsList";
import OrderAddressModal from "./components/OrderAddressModal";
import useDeliveryMethodByTime from "@/src/hooks/Queries/useDeliveryMethodByTime";
import { convertEnglishWeekdayToPersian } from "@/src/utils/Helpers";
import { DayTimePickerDefaultValues } from "@/src/components/Share/DayTimePickerForm";
import SelectDeliveryTimeModal from "./components/SelectDeliveryTimeModal";
import useAreaManagementAddressQuery from "@/src/hooks/Queries/UseAreaManagementAddressQuery";
import { useSelectionStore } from "@/src/store/useSelectionStore";
import { useRestaurantSelection } from "../../useRestaurantSelection";

interface AddressParams {
  delivery_type_id: string;
  address_id?: string;
  branch_id: string;
}

interface Props {
  onSetAddress: (params: AddressParams) => void;
  onSetDiscountCode: (price: number, code: string) => void;
  onSetGateway: (id: number) => void;
  onSetDelivery: (delivery: string, selectedAddressId?: string) => void;
  onAddressResult?: (result: ServiceDetails) => void;
  // onSetTime: (time: string) => void;
  // time: string;
  onBack: VoidFunction;
  address?: Address;
  discountCode: string;
  gateway: number;
  delivery?: DeliveryType;
  chefId: string;
  deliveryLoading: boolean;
  onSetWalletPayment?: (useWallet: boolean) => void;
  useWalletPayment?: boolean;
}

function CardFinal({
  onSetAddress,
  onBack,
  address,
  onAddressResult,
  onSetDiscountCode,
  discountCode,
  gateway,
  onSetGateway,
  chefId,
  onSetDelivery,
  deliveryLoading,
  onSetWalletPayment,
  useWalletPayment = false,
}: Props) {
  const {
    deliveryTime,
    setDeliveryTime,
    selectedDeliveryId,
    selectedAddressId,
    setSelectedDeliveryId,
  } = useRestaurantSelection(chefId);

  const [pickerDefaults, setPickerDefaults] = useState<
    DayTimePickerDefaultValues | undefined
  >();

  const {
    data: deliveryData,
    isLoading: loadingDelivery,
    isSuccess,
    isFetching,
  } = useDeliveryMethodByTime({
    branch_id: chefId,
    day: deliveryTime?.day,
    time: deliveryTime?.time,
  });
  const { authUser } = useAuthStore();
  const { ensureRestaurant, updateSelection, getSelection } =
    useSelectionStore();

  useEffect(() => {
    ensureRestaurant(chefId);
  }, [chefId, ensureRestaurant]);

  const [coords, setCoords] = useState<{
    lat: string | null;
    lng: string | null;
  }>({
    lat: null,
    lng: null,
  });
  useEffect(() => {
    const unsavedAddressRaw = localStorage.getItem("unsavedAddress");
    if (unsavedAddressRaw) {
      try {
        const parsed = JSON.parse(unsavedAddressRaw);
        setCoords({
          lat: parsed.lat ?? null,
          lng: parsed.lng ?? null,
        });
      } catch (e) {
        console.error("Invalid unsavedAddress in localStorage", e);
      }
    }
  }, []);

  const {
    data: areaAddressData,
    error,
    isSuccess: areaAddressIsSuccess,
    isLoading,
  } = useAreaManagementAddressQuery({
    branchId: chefId,
    addressId: authUser?.defaultAddress,
    lat: coords.lat ?? "",
    lng: coords.lng ?? "",
  });

  const { t } = useTranslation();
  const [showAddressList, setShowAddressList] = useState(false);

  const [showModalSelectDeliveryTime, setShowModalSelectDeliveryTime] =
    useState(false);

  const updateDeliveryTime = (day: string, time: string) => {
    // time must be "HH:mm:ss" — build it upstream if needed
    setDeliveryTime({ day, time }); // ✅ persisted via Zustand; survives refresh
  };

  const { data: GatewayData, isLoading: isLoadingGatewayData } =
    useGatewayQuery();

  // useEffect(() => {
  //   if (delivery?.id != null) {
  //     console.log("🚀🚀 ~ useEffect ~ delivery:", delivery);
  //     setSelectedDeliveryId(Number(delivery.id));
  //   }
  // }, [delivery?.id]);

  // Save to localStorage on change
  useEffect(() => {
    const selection = getSelection(chefId);
    const selectedAddressIdMystore = selection?.selectedAddressId;

    if (selectedDeliveryId !== null) {
      updateSelection(chefId, {
        deliveryId: String(selectedDeliveryId),
      });
    }
  }, [selectedDeliveryId]);

  const {
    data: addressOrder,
    isLoading: isLoadingAddressOrder,
    refetchWithParamsOrderAreaAddress,
  } = UseOrderAddressByParamQuery();

  const hasRun = useRef(false);
  const storedAddressOrder = useCartStore((state) => state.addressOrder);

  useEffect(() => {
    if (hasRun.current) {
      return; // If it's already run, exit immediately
    }
    const selection = getSelection(chefId);
    const selectedAddressIdMystore = selection?.selectedAddressId;
    console.log(
      "🚀 ~ CardFinal ~ selectedAddressIdMystore:",
      selectedAddressIdMystore
    );

    if (isSuccess && deliveryData) {
      const hasAddress = deliveryData?.delivery_methods?.find(
        (item) => Number(item.id) === Number(selectedDeliveryId)
      );
      if (
        !hasRun.current &&
        isSuccess && // Ensure API call is complete
        deliveryData &&
        selectedDeliveryId &&
        chefId &&
        authUser?.id
      ) {
        hasRun.current = true; // Prevent future executions
        refetchWithParamsOrderAreaAddress({
          branch_id: Number(chefId),
          user_id: Number(authUser.id),
          delivery_type_id: Number(selectedDeliveryId),
        });

        onSetAddress({
          delivery_type_id: `${selectedDeliveryId}`,
          address_id: `${selectedAddressIdMystore}`,
          branch_id: chefId,
        });
      }
    }
    return () => {
      console.log("Component unmounted - Cleanup ran.");
    };
  }, [isSuccess]); // Run when API fetch status changes

  const handleRadioClick = async (e: React.MouseEvent, option: any) => {
    // Temporarily clear the selected ID to force re-render
    if (selectedDeliveryId !== option.id) {
      setSelectedDeliveryId(option.id);

      if (option.is_address) {
        localStorage.setItem("has_address", "1");
        if (!storedAddressOrder || storedAddressOrder.length === 0) {
          await refetchWithParamsOrderAreaAddress({
            branch_id: Number(chefId),
            user_id: Number(authUser.id),
            delivery_type_id: Number(option.id),
          });
        } else {
          console.log("✅ Skipped refetch — addressOrder exists in store");
        }
        if (selectedAddressId) {
          onSetDelivery(option.id, `${selectedAddressId}`);
        }
      } else {
        localStorage.setItem("has_address", "0");
        onSetDelivery(option.id);
      }
    }
  };

  useEffect(() => {
    if (isSuccess && deliveryData?.next_available) {
      const { day, service_time_from } = deliveryData.next_available;
      const [hourStr, minuteStr] = service_time_from.split(":");

      setPickerDefaults({
        day: day.toLowerCase(),
        hour: parseInt(hourStr),
        minute: parseInt(minuteStr),
      });
    }
  }, [isSuccess, deliveryData]);

  const displayPersianTime = deliveryTime
    ? `${convertEnglishWeekdayToPersian(deliveryTime.day)} ساعت ${deliveryTime.time?.slice(0, 5)}`
    : "زمانی انتخاب نشده است";

  return (
    <div className="flex w-full max-w-2xl  flex-col items-center gap-8 rounded-half  lg:p-12">
      <div className="flex w-full border-b border-black/20 pb-12 ">
        <button onClick={onBack} className="flex items-center gap-1 text-h3  ">
          <ArrowRight2 size="24" color="#000000" variant="Broken" />
          بازگشت و ویرایش سبد خرید
        </button>
      </div>
      <div className="rounded-lg  font-vazir relative mx-auto w-full border text-right">
        <div className="flex items-center justify-between border-b">
          <div className="flex">
            <p className=" py-8 pr-8 text-h5 font-bold">زمان ارسال :</p>
            <div className="flex items-center py-4 text-h5">
              {areaAddressData?.active ? (
                <div>
                  {deliveryTime?.time ? (
                    <div className="space-y-1">
                      <div className="text-h5 font-semibold text-gray-700">
                        <span className="ms-1">
                          {convertEnglishWeekdayToPersian(deliveryTime.day)}
                        </span>{" "}
                        <span className="me-2">ساعت</span>
                        {deliveryTime.time}
                      </div>
                    </div>
                  ) : (
                    <p className="text-base text-red-500 font-semibold">الان</p>
                  )}
                </div>
              ) : (
                <div>
                  <span className="mt-2 flex items-center font-medium">
                    {convertEnglishWeekdayToPersian(
                      areaAddressData?.data?.day as string
                    )}{" "}
                    <span className="mx-2">ساعت</span>
                    {areaAddressData?.data?.service_time_from?.slice(0, 5)}
                  </span>
                </div>
              )}
            </div>
          </div>

          <div className="ms-7">
            <button
              onClick={() => setShowModalSelectDeliveryTime(true)}
              className="text-primary-main ml-4 rounded-half border border-primary px-4 py-3 text-h5 transition-colors duration-200 hover:bg-primary hover:text-white"
            >
              تغییر زمان
            </button>
          </div>
        </div>
        {/* Overlay Loader */}
        {deliveryLoading && (
          <div className="absolute inset-0 z-10 flex items-center justify-center bg-white/50 backdrop-blur-sm">
            <LoaderSpinner />
          </div>
        )}
        <>
          {loadingDelivery ? (
            <div className="flex flex-wrap gap-4 p-8">
              <LoaderSpinner />
            </div>
          ) : !deliveryData?.delivery_methods?.length ? (
            <p className="text-h5 text-primary">
              این {t("restaurant")} هیچ نحوه ارسالی ندارد!
            </p>
          ) : (
            <DeliveryOptionsList
              deliveryData={deliveryData?.delivery_methods}
              selectedDeliveryId={Number(selectedDeliveryId)}
              address={address}
              handleRadioClick={handleRadioClick}
              setShowAddressList={setShowAddressList}
            />
          )}
        </>
      </div>
      <table className="w-full border border-black/20">
        <thead>
          <tr className="border border-black/20 p-8 text-right text-h5">
            <th className="p-8">
              <div className="flex justify-between">
                <p> روش پرداخت</p>
                {authUser?.wallet_balance && authUser.wallet_balance > 0 && (
                  <div className="flex gap-4 items-center justify-between">
                    <label className="relative inline-flex cursor-pointer items-center">
                      <input
                        type="checkbox"
                        className="peer sr-only"
                        checked={useWalletPayment}
                        onChange={(e) => onSetWalletPayment?.(e.target.checked)}
                      />
                      <div className="relative h-12 w-[49px] rounded-full bg-gray-300 peer-checked:bg-Success-main peer-focus:outline-none peer-focus:ring-2 peer-focus:ring-Success-light">
                        <div
                          className={`absolute left-1 top-1 h-10 w-10 rounded-full bg-white transition-transform duration-200 ${useWalletPayment ? "translate-x-12" : "translate-x-0"}`}
                        ></div>
                      </div>
                    </label>
                    <p className="text-h5">
                      استفاده از موجودی کیف پول (اعتبار{" "}
                      {authUser.wallet_balance?.toLocaleString("fa-IR")} تومان)
                    </p>
                  </div>
                )}
              </div>
            </th>
          </tr>
        </thead>
        <tbody>
          <tr className="rounded-base border border-black/20 text-right">
            <td>
              <div className="flex flex-wrap gap-4 p-8">
                {isLoadingGatewayData && <LoaderSpinner />}
                {GatewayData?.map((item) => (
                  <div
                    className="flex flex-col flex-wrap lg:flex-row "
                    key={item.id}
                  >
                    <input
                      type="checkbox"
                      id={`gate_${item.id}`}
                      disabled={!item.isActive}
                      className="peer hidden"
                      onChange={() => onSetGateway(item.id)}
                      checked={Number(gateway) === Number(item.id)}
                    />
                    <label
                      htmlFor={`gate_${item.id}`}
                      className="flex w-full cursor-pointer select-none flex-col gap-1 rounded-base border border-black/20 p-4
                  text-h4 font-bold  transition-colors duration-200 ease-in-out peer-checked:border-primary  peer-checked:text-primary "
                    >
                      <span className="flex items-center gap-4">
                        {item.title}
                        {gateway === item.id && (
                          <TickSquare size="18" className="text-primary" />
                        )}
                      </span>{" "}
                      <span className="text-stroke-primary text-h5">
                        پرداخت آنلاین با تمامی کارت های بانکی
                      </span>
                    </label>
                  </div>
                ))}
              </div>
            </td>
          </tr>
        </tbody>

        {!discountCode && (
          <tfoot>
            <tr className="border border-black/40 p-4 text-right text-h5">
              <th className="p-8">
                <div className="flex w-full flex-col items-center justify-between lg:flex-row">
                  کد تخفیف دارید؟
                  <DiscountForm
                    onVerify={(code, price) => {
                      onSetDiscountCode(price, code);
                    }}
                  />
                </div>
              </th>
            </tr>
          </tfoot>
        )}
      </table>
      <OrderAddressModal
        deliveryData={deliveryData}
        onSetAddress={onSetAddress}
        selectedDeliveryId={Number(selectedDeliveryId)}
        chefId={chefId}
        showAddressList={showAddressList}
        setShowAddressList={setShowAddressList}
        address={address}
      />
      <SelectDeliveryTimeModal
        updateDeliveryTime={updateDeliveryTime}
        setShowModalSelectDeliveryTime={setShowModalSelectDeliveryTime}
        showModalSelectDeliveryTime={showModalSelectDeliveryTime}
        deliverynextAvailable={deliveryData?.next_available}
        isSuccess={isSuccess}
        chefId={chefId}
        isFetching={isFetching}
      />
    </div>
  );
}

export default CardFinal;
