"use client";

import React, { useEffect, useState } from "react";

import { OrderDataVariant } from "@/src/types/general";
import {
  useQueryState,
  parseAsStringEnum,
  parseAsInteger,
} from "next-usequerystate";
import { FOODS_FILTERS } from "@/src/constants/Filters";
import { usePaginatedChefQuery } from "@/src/hooks/Queries";
import ReactPaginate from "react-paginate";
import { ArrowLeft2, ArrowRight2 } from "iconsax-react";
import { useGeolocation } from "@uidotdev/usehooks";
import { FullScreenModal } from "@/src/components/FullScreenModal";
import ChefsItems from "./ChefsItems";

import ChefOrder from "./ChefOrder";
import CategorySlider from "../HomePage/CategorySlider";

function ChefsPage() {
  const [defaultOrder, setDefaultOrder] = useQueryState(
    FOODS_FILTERS.ORDER_BY,
    parseAsStringEnum<OrderDataVariant>(
      Object.values(OrderDataVariant)
    ).withDefault(OrderDataVariant.ASC)
  );


  const [lat, setLat] = useState<string | null>(null);
  const [lng, setLng] = useState<string | null>(null);

  useEffect(() => {
    if (typeof window !== "undefined") {
      const unsavedAddress = localStorage.getItem("unsavedAddress");
      if (unsavedAddress) {
        try {
          const addressData = JSON.parse(unsavedAddress);
          setLat(addressData.lat || null);
          setLng(addressData.lng || null);
        } catch (error) {
          console.error("Error parsing unsavedAddress:", error);
        }
      }
    }
  }, []);

 

  const [defaultCategory] = useQueryState(FOODS_FILTERS.CATEGORY);
  const [title] = useQueryState(FOODS_FILTERS.SEARCH);
  const [page, setPage] = useQueryState(
    FOODS_FILTERS.PAGE,
    parseAsInteger.withDefault(1)
  );

  const { data: Chef, isLoading } = usePaginatedChefQuery({
    order: defaultOrder,
    categoryID: defaultCategory ?? "",
    page,
    title: title ?? "",
    ...(lat ? { lat } : {}),
    ...(lng ? { lng } : {})
  });

  const [filterMenuOpen, setFilterMenuOpen] = useState(false);
  const locationState = useGeolocation();

  return (
    <div className="flex w-full flex-col">
      <div className="lg:container ">
        <div className="flex flex-col items-start justify-center gap-12 lg:gap-20">
          <CategorySlider
            wrapperClassName=" lg:px-0 pt-12 pr-12"
            parentID={defaultCategory ?? ""}
          />

          <div className="container flex w-full items-center justify-center  lg:hidden">
            <button
              onClick={() => setFilterMenuOpen(true)}
              className="hover:text-bold flex w-full items-center justify-center whitespace-nowrap rounded-half border border-primary px-12 py-4 text-h3  text-primary transition-all  hover:bg-error-extraLight"
            >
              فیلترها
            </button>
          </div>
          <div className="container flex w-full flex-col items-center justify-center gap-12  pb-10   lg:px-0 lg:pb-20">
            <ChefsItems
              hasLocationPermission={
                !!locationState.error === false || !locationState.loading
              }
              isLoading={isLoading}
              chefs={Chef?.data ?? []}
              order={defaultOrder}
              onChangeOrder={(type) => {
                setPage(1);
                setDefaultOrder(type);
              }}
            />
            <ReactPaginate
              disableInitialCallback
              className="flex h-max items-center gap-8 text-body"
              previousLabel={
                <ArrowRight2
                  size="16"
                  className="text-primary"
                  variant="Broken"
                />
              }
              onPageChange={({ selected }) => {
                window.scroll({ top: 0, left: 0, behavior: "smooth" });
                setPage(selected + 1);
              }}
              pageRangeDisplayed={3}
              pageCount={Chef?.totalPages ?? 0}
              nextLabel={
                <ArrowLeft2
                  size="16"
                  className="text-primary"
                  variant="Broken"
                />
              }
              renderOnZeroPageCount={() => null}
              initialPage={page - 1}
              pageClassName="flex items-center justify-center size-12 text-black"
              activeClassName="rounded-full bg-secondary text-white"
              forcePage={page - 1}
            />
          </div>
        </div>
      </div>
      <FullScreenModal
        title="فیلترها"
        open={filterMenuOpen}
        onClose={() => setFilterMenuOpen(false)}
      >
        <ChefOrder
          hasLocationPermission={
            !!locationState.error === false || !locationState.loading
          }
          onChangeOrder={(type) => {
            setFilterMenuOpen(false);
            setPage(1);
            setDefaultOrder(type);
          }}
          order={defaultOrder}
        />
      </FullScreenModal>
    </div>
  );
}

export default ChefsPage;
