import Button from "@/src/components/Button";
import { Modal } from "@/src/components/Modal";

import React, { useEffect, useState } from "react";

import Image from "next/image";
import { usePaginatedChefQuery } from "@/src/hooks/Queries";
import { OrderDataVariant } from "@/src/types/general";
import Link from "next/link";
import { useTranslation } from "react-i18next";
import ChefRegistrationForm from "./ChefRegistrationForm";

interface Props {
  title: string;
  btnText: string;
  image: string;
  description: string;
  sellerTitle: string;
}

function ChefRegistration({
  btnText,
  description,
  image,
  title,
  sellerTitle,
}: Props) {
  const [showModal, setShowModal] = useState(false);

    const [lat, setLat] = useState<string | null>(null);
    const [lng, setLng] = useState<string | null>(null);
  
    useEffect(() => {
      if (typeof window !== "undefined") {
        const unsavedAddress = localStorage.getItem("unsavedAddress");
        if (unsavedAddress) {
          try {
            const addressData = JSON.parse(unsavedAddress);
            setLat(addressData.lat || null);
            setLng(addressData.lng || null);
          } catch (error) {
            console.error("Error parsing unsavedAddress:", error);
          }
        }
      }
    }, []);
  
  const { data, isLoading } = usePaginatedChefQuery({
    order: OrderDataVariant.RATE,
    paginate: 6,
    ...(lat ? { lat } : {}),
    ...(lng ? { lng } : {})
  });
  const { t } = useTranslation();
  return (
    <>
      <div className=" container flex h-max w-full flex-col items-center justify-center gap-6 lg:flex-row  lg:justify-between lg:gap-0 lg:px-0  ">
        <div className="flex h-full flex-col  justify-between gap-6 lg:gap-15  ">
          <h2 className="text-h4 font-bold  lg:text-h2   "> {title}</h2>
          <p className="max-w-md  text-h4 text-black/80 lg:text-h2 ">
            {description}
          </p>
          <Button onClick={() => setShowModal(true)} className="w-max">
            {btnText}
          </Button>
        </div>
        <div className=" relative  h-fit w-[280px]  lg:w-[450px] ">
          <Image
            className="size-full"
            src={image}
            width={500}
            height={500}
            alt="aa"
          />
          {!isLoading && data && (
            <div className="bottom-0  right-0 flex max-w-max flex-col gap-4 rounded-[20px] bg-white p-8 shadow-100 lg:absolute lg:gap-8">
              <p className="text-h5 font-bold lg:text-h3">{sellerTitle}</p>
              <div className="flex -space-x-4 rtl:space-x-reverse">
                {data.data.slice(0, 6).map((item) => (
                  <Image
                    key={item.id}
                    className="size-20 rounded-full  "
                    width={80}
                    height={80}
                    alt={item.title}
                    src={item.logo ?? ""}
                  />
                ))}
                <Link
                  className="flex size-20 items-center justify-center
               rounded-full border-2 bg-secondary 
                text-h4 font-medium text-white "
                  href="/chefs"
                >
                  +{data.total}
                </Link>
              </div>
            </div>
          )}
        </div>
      </div>
      <Modal
        title={t("registration")}
        onClose={() => setShowModal(false)}
        open={showModal}
        wrapperClassName="lg:max-w-2xl max-w-lg w-full "
      >
        <ChefRegistrationForm onFormSubmit={() => setShowModal(false)} />
      </Modal>
    </>
  );
}

export default ChefRegistration;
