"use client";

import React, { useEffect, useState } from "react";

import { OrderDataVariant } from "@/src/types/general";
import {
  useQueryState,
  parseAsStringEnum,
  parseAsInteger,
} from "next-usequerystate";
import { FOODS_FILTERS } from "@/src/constants/Filters";
import { usePaginatedFoodsQuery } from "@/src/hooks/Queries";
import ReactPaginate from "react-paginate";
import { ArrowLeft2, ArrowRight2 } from "iconsax-react";
import { useGeolocation } from "@uidotdev/usehooks";
import { FullScreenModal } from "@/src/components/FullScreenModal";
import ProductsItems from "./ProductsItems";

import ProductOrder from "./ProductOrder";
import ProductCategory from "./ProductCategory";

function ProductsPage() {
  const [defaultOrder, setDefaultOrder] = useQueryState(
    FOODS_FILTERS.ORDER_BY,
    parseAsStringEnum<OrderDataVariant>(
      Object.values(OrderDataVariant)
    ).withDefault(OrderDataVariant.ASC)
  );


  const [lat, setLat] = useState<string | null>(null);
  const [lng, setLng] = useState<string | null>(null);

  useEffect(() => {
    if (typeof window !== "undefined") {
      const unsavedAddress = localStorage.getItem("unsavedAddress");
      if (unsavedAddress) {
        try {
          const addressData = JSON.parse(unsavedAddress);
          setLat(addressData.lat || null);
          setLng(addressData.lng || null);
        } catch (error) {
          console.error("Error parsing unsavedAddress:", error);
        }
      }
    }
  }, []);

  const [selectedID, setSelectedID] = useState<string | undefined>(undefined);

  const [defaultCategory, setDefaultCategory] = useQueryState(
    FOODS_FILTERS.CATEGORY
  );

  const [title] = useQueryState(FOODS_FILTERS.SEARCH);
  const [page, setPage] = useQueryState(
    FOODS_FILTERS.PAGE,
    parseAsInteger.withDefault(1)
  );

  const { data: Product, isLoading, isFetching } = usePaginatedFoodsQuery({
    order: defaultOrder,
    categoryID: defaultCategory ?? "",
    page,
    title: title ?? "",
    ...(lat ? { lat } : {}),
    ...(lng ? { lng } : {})
  });

  const [filterMenuOpen, setFilterMenuOpen] = useState(false);
  const locationState = useGeolocation();

  useEffect(() => {
    if (selectedID) {
      setDefaultCategory(selectedID);
      setFilterMenuOpen(false)
    }

  }, [selectedID])


  return (
    <div className="flex w-full flex-col">
      <div className="lg:container ">
        <div className="flex flex-col items-start justify-center gap-12 lg:gap-20">


          <div className="container mt-8 flex w-full items-center justify-center  lg:hidden">
            <button
              onClick={() => setFilterMenuOpen(true)}
              className="hover:text-bold flex w-full items-center justify-center whitespace-nowrap rounded-half border border-primary px-12 py-4 text-h3  text-primary transition-all  hover:bg-error-extraLight"
            >
              فیلترها
            </button>
          </div>
          <div className="container flex w-full flex-col items-center justify-center gap-12  pb-10   lg:px-0 lg:pb-20">
            <ProductsItems
              defaultCategory={defaultCategory ?? ""}
              onChangeCategories={(categoryId) => {
                setPage(1);
                setDefaultCategory(categoryId);
              }}
              hasLocationPermission={
                !!locationState.error === false || !locationState.loading
              }
              isLoading={isFetching}
              chefs={isFetching ? [] : Product?.data ?? []}
              order={defaultOrder}
              onChangeOrder={(type) => {
                setPage(1);
                setDefaultOrder(type);
              }}
            />
            <ReactPaginate
              disableInitialCallback
              className="flex h-max items-center gap-8 text-body text-white "
              previousLabel={
                <ArrowRight2
                  size="16"
                  className="text-primary"
                  variant="Broken"
                />
              }
              onPageChange={({ selected }) => {
                window.scroll({ top: 0, left: 0, behavior: "smooth" });
                setPage(selected + 1);
              }}
              pageRangeDisplayed={3}
              pageCount={Product?.totalPages ?? 0}
              nextLabel={
                <ArrowLeft2
                  size="16"
                  className="text-primary"
                  variant="Broken"
                />
              }
              renderOnZeroPageCount={() => null}
              initialPage={page - 1}
              pageClassName=" flex items-center justify-center text-black size-12"
              activeClassName="rounded-full bg-secondary text-white"
              forcePage={page - 1}
            />
          </div>
        </div>
      </div>
      <FullScreenModal
        title="فیلترها"
        open={filterMenuOpen}
        onClose={() => setFilterMenuOpen(false)}
      >
        <ProductOrder
          hasLocationPermission={
            !!locationState.error === false || !locationState.loading
          }
          onChangeOrder={(type) => {
            setFilterMenuOpen(false);
            setPage(1);
            setDefaultOrder(type);
          }}
          order={defaultOrder}
        />
        <ProductCategory
          setSelectedID={setSelectedID}
          selectedID={selectedID}
        />

      </FullScreenModal>
    </div>
  );
}

export default ProductsPage;
