import { create } from "zustand";

import { CartItem, DeliveryTime, DeliveryType } from "../types/Cart";
import { AdrressProp } from "../hooks/Queries/UseOrderAddressByParamQuery";

const initialCard: CartItem[] = [];

interface CartStore {
  cardItems: CartItem[] | [];
  synced: boolean;
  chefId: string;
  delivery?: DeliveryType;
  deliveryTime?: DeliveryTime;
  setDelivery: (delivery: DeliveryType) => void;
  setDeliveryTime: (day: string, time: string) => void;
  setChefId: (id: string) => void;
  setSynced: (state: boolean) => void;
  count: () => number;
  addToCard: (food: CartItem) => void;
  removeFromCard: (varietyId: string) => void;
  itemCount: (varietyId: string) => number;
  resetCard: () => void;
  updateCart: (newCart: CartItem[]) => void;
  addressOrder: AdrressProp[];
  setAddressOrder: (data: AdrressProp[]) => void;
}

const UseCartStore = create<CartStore>()((set, get) => ({
  addressOrder: [],
  setAddressOrder: (data: AdrressProp[]) => set(() => ({ addressOrder: data })),

  cardItems: initialCard,
  setDelivery: (delivery: DeliveryType) =>
    set(() => ({
      delivery,
    })),
  setDeliveryTime: (day: string, time: string) =>
    set(() => ({
      deliveryTime: {
        day,
        time,
      },
    })),
  chefId: "",
  setChefId: (id: string) =>
    set(() => ({
      chefId: id,
    })),
  synced: false,
  count: () => {
    const { cardItems } = get();
    return cardItems.length;
  },
  resetCard: () =>
    set(() => ({
      cardItems: initialCard,
      chefId: "",
      synced: false,
      delivery: undefined,
    })),
  addToCard: (food) => {
    const { cardItems } = get();
    const updatedCart = [...cardItems];
    const productIndex = cardItems.findIndex(
      (item) => item.variety.varietyID === food.variety.varietyID
    );

    if (productIndex !== -1) {
      // Update existing item
      updatedCart[productIndex] = {
        ...updatedCart[productIndex],
        count: updatedCart[productIndex].count + (food?.count ?? 1),
      };
    } else {
      // Add new item
      updatedCart.push({ ...food, count: food?.count ?? 1 });
    }
    
    set({
      cardItems: updatedCart,
      synced: false,
      ...(cardItems.length === 0 && {
        chefId: food.food.chefID,
      }),
    });
  },
  removeFromCard: (id) => {
    const { cardItems } = get();
    const updatedCart = [...cardItems];
    const productIndex = cardItems.findIndex(
      (item) => item.variety.varietyID === id
    );
    if (productIndex !== -1) {
      updatedCart[productIndex] = {
        ...updatedCart[productIndex],
        count: updatedCart[productIndex].count - 1,
      };
    }
    const newItems = updatedCart.filter((item) => item.count);
    set({
      cardItems: newItems,
      synced: false,
      ...(newItems.length === 0 && {
        chefId: "",
        deliveryTime: undefined,
      }),
    });
  },
  itemCount: (id) => {
    const { cardItems } = get();
    const item = cardItems.find((item) => item.variety.varietyID === id);
    const count = item?.count ?? 0;
    
    // Ensure count is always a valid number
    return typeof count === 'number' && count >= 0 ? count : 0;
  },
  updateCart: (newCart: CartItem[]) =>
    set(() => ({
      cardItems: newCart,
    })),
  setSynced: (syncState: boolean) =>
    set(() => ({
      synced: syncState,
    })),
}));

export default UseCartStore;
