import { useUpload } from "@/src/hooks";
import { DocumentUpload } from "iconsax-react";
import React, { useRef, useState } from "react";
import { MAX_FILE_SIZE, allowFormats } from "@/src/configs";
import { useSnackbar } from "notistack";
import LoaderSpinner from "../LoaderSpinner";

interface Props {
  onUploadComplete: (path: string) => void;
}

function FileUploader({ onUploadComplete }: Props) {
  const [selectedFile, setSelectedFile] = useState<File | undefined>();
  const inputRef = useRef<HTMLInputElement>(null);
  const { isLoading, uploadFile } = useUpload({
    onUploadComplete: () => {
      setSelectedFile(undefined);
      if (inputRef.current) {
        inputRef.current.files = null;
      }
    },
  });
  const { enqueueSnackbar } = useSnackbar();
  const uploadHandler = async (files?: FileList) => {
    if (files && files[0]) {
      const file = files[0];
      const fileExtension = file.type.split("/").pop();
      if (file.size > MAX_FILE_SIZE) {
        enqueueSnackbar("حداکثر تا 50 مگابایت قابل بارگذاری است", {
          variant: "error",
        });
        return false;
      }
      if (fileExtension && allowFormats.includes(fileExtension.toLowerCase())) {
        const path = await uploadFile(file, file.name);
        onUploadComplete(path);
        setSelectedFile(file);
      } else {
        enqueueSnackbar("فرمت فایل نا معتبر", {
          variant: "error",
        });
      }
    }
    return true;
  };
  return (
    <div className="">
      <label
        className={`border-stroke-primary flex w-max max-w-full  cursor-pointer items-center gap-4 rounded-base border ${
          !isLoading ? "bg-white" : "bg-gray-600/50 text-white"
        } px-12 py-4 text-h5 text-primary `}
        htmlFor="file_input"
      >
        {isLoading ? <LoaderSpinner /> : <DocumentUpload />}
        {selectedFile ? `فایل انتخابی : ${selectedFile.name}` : ""}

        {!isLoading && !selectedFile && "تصویر یا فایل مورد نظر بارگذاری کنید"}
        {isLoading && "درحال بارگذاری..."}
      </label>
      <input
        ref={inputRef}
        className="hidden"
        id="file_input"
        type="file"
        disabled={isLoading}
        onChange={(e) => uploadHandler(e.target.files ?? undefined)}
      />
      <p className="text-h5 text-text-secondary">فرمت های مجاز : jpg png txt</p>
    </div>
  );
}

export default FileUploader;
