import React from "react";
import { numberWithCommas } from "@/src/utils/Helpers";
import { Variety } from "@/src/types/food";
import { useCartStore } from "@/src/store";
import { twMerge } from "tailwind-merge";
import AddCardButton from "../AddCardButton";
import NextImg from "../NextImage";
import { useCartItemCount } from "@/src/hooks/useCartItemCount";

interface Props {
  title: string;

  image: string;
  onAddCard: (variety: Variety) => void;
  onRemoveCard: (varietyID: string) => void;
  onSelect?: VoidFunction;
  available?: boolean;
  className?: string;
  description?: string
  titleClassName?: string;
  varieties: Variety[];
}

function MiniProductCard({
  image,
  title,
  description,
  onAddCard,
  onRemoveCard,
  className,
  available = true,
  titleClassName,
  onSelect,
  varieties,
}: Props) {

  return (
    <div
      className={twMerge(
        "flex flex-col items-center justify-center gap-4 border  border-black/10 px-8 py-4",
        className
      )}
    >
      <div
        className="flex w-full cursor-pointer sm:flex-col items-center gap-4"
        onClick={onSelect}
      >
        <div className="flex w-full  items-center justify-center ">
          <NextImg
            imgStyles={{placeSelf:'anchor-center'}}
            src={image}
            width={170}
            height={170}
            imgWrapperClassName="rounded-half overflow-hidden"
            imgClassName="!object-contain w-[100px] md:w-full "
            alt={title}
          />
        </div>
        <div className="flex flex-col">
          <p
            className={twMerge(
              "w-full max-w-80 truncate  text-right text-h4 ",
              titleClassName
            )}
          >
            {title}

          </p>
          <p className={twMerge(
              "w-full max-w-80 lg:hidden truncate mt-4 text-right text-h5 ",
              titleClassName
            )}>{description}</p>
        </div>
      </div>

      {varieties.map((item) => (
        <div
          key={item.varietyID}
          className="flex w-full  items-center justify-between h-[20px]"
        >
          <div>
            <div className="flex flex-wrap items-center gap-2">
              {+item.price !== 0 && (
                <>
                  {item.discount?.percent !== 0 && (
                    <>
                      <div
                        className="flex h-10 min-w-10 items-center justify-center rounded-base bg-primary  text-center
                           text-h6  text-white"
                      >
                        {item.discount?.percent} %
                      </div>
                      <span className="text-h5  font-light text-secondary line-through ">
                        <span>{numberWithCommas(item.price)}</span>
                      </span>
                    </>
                  )}
                  <p className="text-h3 font-bold">
                    {numberWithCommas(item.discount?.discountedPrice ?? 0)}{" "}
                    <span className="text-h5 font-normal">تومان</span>
                  </p>
                </>
              )}
            </div>
          </div>

          {available && (
            <MiniProductCardButton
              item={item}
              onAddCard={() => onAddCard(item)}
              onRemoveCard={() => onRemoveCard(item.varietyID)}
            />
          )}
        </div>
      ))}
    </div>
  );
}

// Separate component to handle cart count reactivity for mini product cards
function MiniProductCardButton({ 
  item, 
  onAddCard, 
  onRemoveCard 
}: { 
  item: Variety; 
  onAddCard: () => void; 
  onRemoveCard: () => void; 
}) {
  const productInCart = useCartItemCount(item.varietyID);
  
  return (
    <AddCardButton
      isLoading={false}
      onAddCard={onAddCard}
      onRemoveCard={onRemoveCard}
      productInCart={productInCart}
      limit={item.maxOrderLimit}
      quantity={item.quantity}
    />
  );
}

export default MiniProductCard;
