import React from "react";
import { numberWithCommas } from "@/src/utils/Helpers";
import { Variety } from "@/src/types/food";
import { twMerge } from "tailwind-merge";
import AddCardButton from "../AddCardButton";
import NextImg from "../NextImage";
import { useCartItemCount } from "@/src/hooks/useCartItemCount";

interface Props {
  title: string;
  Summary: string;
  image: string;
  onAddCard: (variety: Variety) => void;
  onRemoveCard: (varietyID: string) => void;
  onSelect?: VoidFunction;
  available?: boolean;
  className?: string;
  titleClassName?: string;
  varieties: Variety[];
  isLoadingCart: boolean;
}

function ProductCard({
  image,
  isLoadingCart,
  Summary,
  title,
  onAddCard,
  onRemoveCard,
  className,
  available = true,
  titleClassName,
  onSelect,
  varieties,
}: Props) {
  const img = (typeof image === "string" ? image.trim() : "");
  const hasImage =
    img.length > 0 &&
    img !== "null" &&
    img !== "undefined" &&
    !img.toLowerCase().includes("placeholder");


  return (
    <div
      className={twMerge(
        "flex flex-col gap-10 rounded-[5px] border border-gray-200 bg-white p-10",
        className
      )}
    >
      <div
        className={twMerge("flex w-full cursor-pointer gap-4", hasImage ? "items-center" : "items-start")}
        onClick={onSelect}
      >
        {hasImage && (
          <div className="min-w-[70px] max-w-fit">
          <NextImg
            src={image}
            width={67}
            imgClassName="rounded-base"
            height={67}
            alt="img"
          />
        </div>
        )}
        <div className="w-full overflow-hidden">
          <p
            className={twMerge(
              "w-full md:max-w-[175px] truncate text-h3 ",
              titleClassName
            )}
          >
            {title}
          </p>

          <p className="line-clamp-2 w-full text-[11px] leading-5 text-text-secondary">
            {Summary}
          </p>
        </div>
      </div>
      {varieties.map((item) => (
        <div
          key={item.varietyID}
          className="flex w-full  items-center justify-between"
        >
          <div>
            <span className="text-h5">{item.title}</span>
            <div className="flex items-center gap-6">
              {+item.price !== 0 && (
                <>
                  {item.discount?.percent !== 0 && (
                    <>
                      <div
                        className="flex h-7 min-w-7 items-center justify-center rounded-[5px] bg-red-500 text-center text-[9px] font-semibold text-white"
                      >
                        {item.discount?.percent} %
                      </div>
                      <span className="text-h5  font-light text-secondary line-through ">
                        <span>{numberWithCommas(item.price)}</span>
                      </span>
                    </>
                  )}
                  <p className="text-h3 font-bold ">
                    {numberWithCommas(item.discount?.discountedPrice ?? 0)}{" "}
                    <span className="text-h5 font-normal">تومان</span>
                  </p>
                </>
              )}
            </div>
          </div>

          {available && (
            <ProductCardAddButton
              item={item}
              onAddCard={() => onAddCard(item)}
              onRemoveCard={() => onRemoveCard(item.varietyID)}
              isLoading={isLoadingCart}
            />
          )}
        </div>
      ))}
    </div>
  );
}

// Separate component to handle cart count reactivity
function ProductCardAddButton({ 
  item, 
  onAddCard, 
  onRemoveCard, 
  isLoading 
}: { 
  item: Variety; 
  onAddCard: () => void; 
  onRemoveCard: () => void; 
  isLoading: boolean; 
}) {
  const productInCart = useCartItemCount(item.varietyID);
  
  return (
    <AddCardButton
      onAddCard={onAddCard}
      onRemoveCard={onRemoveCard}
      productInCart={productInCart}
      limit={item.maxOrderLimit || 10}
      quantity={item.quantity || 10}
      isLoading={isLoading}
    />
  );
}

export default ProductCard;
