import { FormProvider, Select } from "@/src/components/RHF";
import React, { useMemo, useState } from "react";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as Yup from "yup";
import Button from "@/src/components/Button";
import { useCart } from "@/src/hooks";
import { useSnackbar } from "notistack";
import { useOrderAddressQuery } from "@/src/hooks/Queries";
import { FormOption } from "@/src/types/general";
import { ArrowLeft2 } from "iconsax-react";
import { Modal } from "@/src/components/Modal";
import { useCartStore } from "@/src/store";
import { useTranslation } from "react-i18next";
import AddressForm from "../../Profile/profileTabs/Address/AddressForm";

interface FormFields {
  address: FormOption;
}

const defaultValues: FormFields = {
  address: {
    label: "",
    value: "",
  },
};
interface Props {
  onAddAddress: VoidFunction;
}
function CartAddressForm({ onAddAddress }: Props) {
  const addressSchema = useMemo(
    () =>
      Yup.object().shape({
        address: Yup.object().shape({
          label: Yup.string().required("لطفا ادرس خود را وارد نمایید "),
          value: Yup.string()
            .required("لطفا ادرس خود را وارد نمایید ")
            .matches(/^[0-9]*$/, "لطفا ادرس خود را بصورت درست وارد کنید "),
        }),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );
  const [showAddAddress, setShowAddAddressModal] = useState(false);
  const methods = useForm<FormFields>({
    defaultValues,
    resolver: yupResolver(addressSchema),
    mode: "onChange",
  });

  const {
    handleSubmit,
    formState: { isValid, isSubmitting },
    reset,
  } = methods;
  const { setAddress } = useCart();
  const { enqueueSnackbar } = useSnackbar();
  const { data, refetch, isLoading } = useOrderAddressQuery();
  const delivery = useCartStore((state) => state.delivery);

  const onSubmit = handleSubmit(async (values) => {
    try {
      await setAddress({
        delivery_type_id: delivery?.id,
        address_id: values.address.value,
      });
      enqueueSnackbar("تغییرات با موفقیت انجام شد", {
        variant: "success",
      });
      reset();
      onAddAddress();
    } catch (error) {
      console.log(error,'errpr');
      enqueueSnackbar(error? error.message: 'خطا در ثبت اطلاعات', {
        variant: "error",
      });
    }
  });
  const { t } = useTranslation();

  return (
    <>
      <div className="flex w-full  flex-col  px-16 py-12">
        <FormProvider methods={methods} onSubmit={onSubmit}>
          <div className=" flex w-full flex-col gap-8">
            <div className="grid grid-cols-1   ">
              <Select
                fullWidth
                options={data}
                name="address"
                label="عنوان آدرس*"
                placeholder="انتخاب کنید*"
              />{" "}
              {data && !data.length && !isLoading && (
                <span className="my-4 text-small font-light text-secondary">
                  شما آدرسی در محدوده {t("restaurant")} ندارید !
                </span>
              )}
            </div>
            <Button
              fullWidth
              type="submit"
              isLoading={isSubmitting}
              disabled={!isValid}
            >
              ثبت تغییرات
            </Button>
            <button
              onClick={() => setShowAddAddressModal(true)}
              className="flex  items-center gap-4 text-h5 text-text-link "
            >
              افزودن آدرس جدید
              <ArrowLeft2 size="16" color="#007AFF" variant="Broken" />
            </button>
          </div>
        </FormProvider>
      </div>
      <Modal
        title="افزودن آدرس"
        onClose={() => setShowAddAddressModal(false)}
        open={showAddAddress}
      >
        <AddressForm
          onAddAddress={() => {
            refetch();
            setShowAddAddressModal(false);
          }}
        />
      </Modal>
    </>
  );
}

export default CartAddressForm;
