"use client";

import React, { useState } from "react";

import { OrderDataVariant } from "@/src/types/general";
import {
  useQueryState,
  parseAsStringEnum,
  parseAsInteger,
  parseAsString,
  parseAsBoolean,
} from "next-usequerystate";
import { FOODS_FILTERS } from "@/src/constants/Filters";

import ReactPaginate from "react-paginate";
import {
  ArrowLeft2,
  ArrowRight2,
  DocumentFilter,
  HambergerMenu,
} from "iconsax-react";
import { useDebouncedCallback } from "use-debounce";
import { usePaginatedFoodsQuery } from "@/src/hooks/Queries";

import { useGeolocation, useMediaQuery } from "@uidotdev/usehooks";
import { FullScreenModal } from "@/src/components/FullScreenModal";
import { convertOrderToPersian } from "@/src/utils/Helpers";
import NextImg from "@/src/components/NextImage";
import emptyImage from "@/public/assets/images/default/no-order.svg";
import FoodsSideBar from "./FoodsSideBar";
import FoodsItems from "./FoodsItems";

import ChefOrder from "../Chefs/ChefOrder";
import FoodFilters from "./FoodFilters";

function FoodsPage() {
  const [defaultOrder, setDefaultOrder] = useQueryState(
    FOODS_FILTERS.ORDER_BY,
    parseAsStringEnum<OrderDataVariant>(
      Object.values(OrderDataVariant)
    ).withDefault(OrderDataVariant.DESC)
  );
  const [searchText, setSearchText] = useQueryState(
    FOODS_FILTERS.SEARCH,
    parseAsString.withDefault("")
  );
  const [defaultCategory, setDefaultCategory] = useQueryState(
    FOODS_FILTERS.CATEGORY
  );
  const [page, setPage] = useQueryState(
    FOODS_FILTERS.PAGE,
    parseAsInteger.withDefault(1)
  );
  const [showDiscount, setShowDiscount] = useQueryState(
    FOODS_FILTERS.HAS_DISCOUNT,
    parseAsBoolean.withDefault(false)
  );
  const debounceText = useDebouncedCallback((value) => {
    setPage(1);
    setSearchText(value);
  }, 500);
  const locationState = useGeolocation();
  const { data: foods, isLoading } = usePaginatedFoodsQuery({
    order: defaultOrder,
    categoryID: defaultCategory ?? "",
    page,
    title: searchText,
    hasDiscount: showDiscount,
    location: {
      lat: locationState.latitude?.toString() ?? "",
      lon: locationState.longitude?.toString() ?? "",
    },
  });
  const isDesktop = useMediaQuery("only screen and (min-width: 1024px)");
  const [filterMenuOpen, setFilterMenuOpen] = useState(false);
  return (
    <div className="flex w-full flex-col">
      {/* <ChefSlider type="product_search" /> */}
      <div className="w-full lg:container">
        <div className="flex w-full flex-col items-center justify-center gap-1 py-2  lg:flex-row lg:items-start lg:gap-3">
          {isDesktop ? (
            <FoodsSideBar
              defaultCategory={defaultCategory ?? ""}
              onChangeCategories={(categoryId) => {
                setPage(1);
                setDefaultCategory(categoryId);
              }}
              showDiscount={showDiscount}
              onDiscountSwitch={(newState) => {
                setPage(1);
                setShowDiscount(newState);
              }}
            />
          ) : (
            <div className="flex w-full gap-1 text-h5 text-white">
              <button
                onClick={() => setFilterMenuOpen(true)}
                className="flex w-full items-center gap-1 bg-primary px-2 py-1"
              >
                <DocumentFilter size="14" color="#FFFFFF" variant="Broken" />
                فیلترها
              </button>{" "}
              <button
                onClick={() => setFilterMenuOpen(true)}
                className="flex w-full items-center gap-1 bg-primary px-2 py-1"
              >
                <HambergerMenu size="14" color="#FFFFFF" variant="Broken" />
                نمایش بر اساس: {convertOrderToPersian(defaultOrder)}
              </button>
            </div>
          )}

          <div className="flex w-full flex-col items-center justify-center gap-2 ">
            <FoodFilters
              hasLocationPermission={
                !!locationState.error === false || !locationState.loading
              }
              onSearch={debounceText}
              onChangeOrder={(type) => {
                setPage(1);
                setDefaultOrder(type);
              }}
              order={defaultOrder}
            />
            {!isLoading && !foods?.total && (
              <div className="flex flex-col items-center justify-center gap-3 py-4  text-white">
                <NextImg
                  src={emptyImage}
                  alt="no-address"
                  width={115}
                  height={140}
                />
                <p className="text-h3 ">غذایی یافت نشد !</p>
              </div>
            )}
            <FoodsItems isLoading={isLoading} foods={foods?.data ?? []} />

            <ReactPaginate
              disableInitialCallback
              className="flex h-max items-center gap-2 text-body text-white "
              previousLabel={
                <ArrowRight2
                  size="16"
                  className="text-primary"
                  variant="Broken"
                />
              }
              onPageChange={({ selected }) => {
                window.scroll({ top: 0, left: 0, behavior: "smooth" });
                setPage(selected + 1);
              }}
              pageRangeDisplayed={3}
              pageCount={foods?.totalPages ?? 0}
              nextLabel={
                <ArrowLeft2
                  size="16"
                  className="text-primary"
                  variant="Broken"
                />
              }
              renderOnZeroPageCount={() => null}
              initialPage={page - 1}
              pageClassName=" flex items-center justify-center w-3 h-3 "
              activeClassName="rounded-full bg-secondary"
              forcePage={page - 1}
            />
          </div>
        </div>
      </div>
      <FullScreenModal
        title="فیلترها"
        open={filterMenuOpen}
        onClose={() => setFilterMenuOpen(false)}
      >
        <FoodsSideBar
          showDiscount={showDiscount}
          onDiscountSwitch={(newState) => {
            setPage(1);
            setShowDiscount(newState);
          }}
          defaultCategory={defaultCategory ?? ""}
          onChangeCategories={(categoryId) => {
            setFilterMenuOpen(false);
            setPage(1);
            setDefaultCategory(categoryId);
          }}
        />
        <ChefOrder
          hasLocationPermission={
            !!locationState.error === false || !locationState.loading
          }
          onChangeOrder={(type) => {
            setFilterMenuOpen(false);
            setPage(1);
            setDefaultOrder(type);
          }}
          order={defaultOrder}
        />
      </FullScreenModal>
    </div>
  );
}

export default FoodsPage;
