import dynamic from "next/dynamic";
import Image from "next/image";
import { useMemo, useState } from "react";
import carImage from "@/public/assets/images/icons/car.svg";
import zingImage from "@/public/assets/images/icons/zing.svg";
import headphoneImage from "@/public/assets/images/icons/headphone.svg";
import { Order } from "@/src/types/Cart";
import { AuthUser } from "@/src/types/auth";
import Link from "next/link";
import UseChefCenterQuery from "@/src/hooks/Queries/UseChefCenterQuery";

type CartItemShape = {
  id: string | number;
  count: number | string;
  food?: { title?: string };
  variety?: {
    title?: string;
    price?: number | string;
    discount?: { discountedPrice?: number | string };
    product?: { image?: string };
  };
  extras?: { title: string }[];
};

const numberWithCommas = (x?: number) =>
  typeof x === "number"
    ? x.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",")
    : "0";

// Leaflet/Map component (SSR-safe)
const StaticMap = dynamic(
  () => import("@/src/components/NeshanMapContainer/StaticNeshanMap"),
  { ssr: false }
);

type OrderTrackingProps = {
  order:
    | (Order & {
        user: AuthUser;
        paymentCode: string;
      })
    | undefined;
};

function rials(amount: number) {
  return `${numberWithCommas(amount)} تومان`;
}

function formatFaDate(input: string | Date) {
  const d = typeof input === "string" ? new Date(input) : input;
  try {
    return new Intl.DateTimeFormat("fa-IR", {
      year: "numeric",
      month: "long",
      day: "numeric",
    }).format(d);
  } catch {
    return d.toLocaleDateString("fa-IR");
  }
}

export default function OrderTrackingPage({ order }: OrderTrackingProps) {
  const [expanded, setExpanded] = useState(false);
  const clip = 3; // how many items to show when collapsed

  const chefId = order?.chef?.id; // you set this in your mapper
  const { data: chefCenter } = UseChefCenterQuery(chefId);

  const center: [number, number] = chefCenter
    ? [chefCenter.lat, chefCenter.lng]
    : [15.6979, 11.3362]; // fallback

  const { items, visibleItems, hiddenCount } = useMemo(() => {
    const carts: CartItemShape[] = (order?.cart as any[]) ?? [];
    const items = carts.map((c) => {
      const title = c.variety?.title ?? c.food?.title ?? "";
      const qty = Number(c.count ?? 0);
      const price =
        Number(c.variety?.discount?.discountedPrice ?? 0) ||
        Number(c.variety?.price ?? 0);
      const imageUrl = c.variety?.product?.image;
      const options = c.extras?.map((e) => e.title) ?? [];
      return { id: c.id, title, qty, price, imageUrl, options };
    });

    if (expanded) return { items, visibleItems: items, hiddenCount: 0 };
    const hiddenCount = Math.max(items.length - clip, 0);
    return { items, visibleItems: items.slice(0, clip), hiddenCount };
  }, [order?.cart, expanded]);

  const n = (v: any) => Number(v ?? 0);

  // derive pricing from API `order`
  const subtotal = Number(order?.pricing?.subtotal ?? 0);
  const delivery = Number(order?.pricing?.delivery ?? 0);
  const total = Number(order?.pricing?.total ?? subtotal + delivery);

  const restaurantName = chefCenter?.title ?? "";

  return (
    <main
      dir="rtl"
      className="mx-auto w-full max-w-7xl px-4 py-6 sm:px-6 lg:px-8"
    >
      {/* Map */}
      <section className="rounded-2xl overflow-hidden bg-white">
        <div className="h-[288px] w-full sm:h-[384px]">
          <StaticMap center={center} marker={center} />
        </div>
        <div className="mt-10 flex items-center justify-between">
          {/* Left icon */}
          <Image
            src={zingImage}
            alt="a"
            className="mt-0.5 h-12 w-12 shrink-0 text-gray-500 "
          />

          {/* The line between icons */}
          <div className="relative mx-3 h-[2px] flex-1 overflow-hidden bg-gray-300">
            {/* right 300px in red (clamped so it won't overflow on small widths) */}
            <span
              className="absolute right-0 top-0 h-full bg-[#bf1c1c]"
              style={{ width: "min(300px, 100%)" }}
            />
          </div>

          {/* Right icon */}
          <Image
            src={carImage}
            alt="a"
            className="mt-0.5 h-12 w-12 shrink-0 text-gray-500 "
          />
        </div>
        {/* Toolbar / Status line */}
        <div className="mt-7 flex flex-col gap-4 p-4 sm:flex-row sm:items-start sm:justify-between">
          <div className="text-xs sm:text-sm flex flex-col items-start gap-2">
            {order?.status === "wating_payment" && (
              <>
                <p className="text-[14px] text-gray-600">
                  سفارش در انتظار پرداخت است.
                </p>
                <p className="mt-6 text-[11px] text-gray-600">
                  پس از پرداخت، تایید رستوران انجام می‌شود.
                </p>
              </>
            )}

            {order?.status === "current" && (
              <>
                <p className="text-[14px] text-gray-600">
                  سفارش در انتظار تایید{" "}
                  <span className="me-2 px-1 text-primary [unicode-bidi:plaintext]">
                    رستوران {restaurantName}
                  </span>
                  می‌باشد.
                </p>
                <p className="mt-6 text-[11px] text-gray-600">
                  پس از این فرآیند آماده‌سازی شروع می‌شود
                </p>
              </>
            )}

            {order?.status === "delivered" && (
              <>
                <p className="text-[14px] text-gray-600">سفارش شما تحویل شد.</p>
                <p className="mt-6 text-[11px] text-gray-600">
                  از خرید شما سپاسگزاریم.
                </p>
              </>
            )}
          </div>

          <div className="mt-5 flex items-center gap-4">
            <Link
              href="/profile?profileTab=tickets"
              className="text-sm text-blue-600 hover:text-blue-700 inline-flex items-center gap-2"
            >
              <Image
                src={headphoneImage}
                alt="پشتیبانی"
                className="h-10 w-10"
              />
              <span className="justify-end text-h5 text-[#157bfb] sm:text-h5 lg:text-h4">
                ارتباط با پشتیبانی
              </span>
            </Link>
          </div>
        </div>
      </section>
      <div className="my-10 border-t border-gray-200 ps-15 pt-10">
        <h2 className="text-lg  font-bold">جزئیات سفارش</h2>
      </div>

      {/* Main Grid */}
      <section className="mt-6 grid grid-cols-1 gap-18 lg:grid-cols-12">
        {/* Details + Items (right in desktop) */}
        <section className="rounded-2xl border-t bg-white pt-6 lg:col-span-8">
          {/* Header */}
          <div className="mb-4 flex flex-wrap items-start justify-between gap-4 pb-4">
            <div className="text-sm grid grid-cols-3 gap-x-24 gap-y-1">
              <div className="flex flex-col items-start gap-2 text-gray-600">
                <span className="text-h5 sm:text-h5 lg:text-h4">تاریخ:</span>
                <strong className="text-h5 text-black lg:text-[12px]">
                  {order?.date}
                </strong>
              </div>
              <div className="flex flex-col items-start gap-2 text-gray-600">
                <span className="text-h5 sm:text-h5 lg:text-h4">مبلغ:</span>
                <strong className="text-h5 text-black lg:text-[12px]">
                  {rials(subtotal)}
                </strong>
              </div>
              <div className="flex flex-col items-start gap-2 text-gray-600">
                <span className="text-h5 sm:text-h5 lg:text-h4">
                  هزینه ارسال:
                </span>
                <strong className="text-h5 text-black lg:text-[12px]">
                  {rials(delivery)}
                </strong>
              </div>
            </div>
          </div>
          {/* Items */}
          <ul className="mt-12">
            {visibleItems.map((it) => (
              <li
                key={it.id}
                className="flex items-center justify-between py-7"
              >
                <div className="flex min-w-0 items-center gap-3">
                  <div className="rounded-lg relative h-10 w-10 overflow-hidden border border-gray-200 bg-gray-50">
                    {it.imageUrl ? (
                      <Image
                        src={it.imageUrl}
                        alt={it.title || "item"}
                        fill
                        className="object-cover"
                      />
                    ) : (
                      <div className="text-xl flex h-full w-full items-center justify-center">
                        🍔
                      </div>
                    )}
                  </div>

                  <div className="min-w-0">
                    <p className="text-gray-900 truncate text-h5 lg:text-h4">
                      {it.title}
                    </p>
                    {!!it.options.length && (
                      <p className="truncate text-h5 text-gray-500 lg:text-h4">
                        {it.options.join("، ")}
                      </p>
                    )}
                  </div>
                </div>

                <div className="text-sm flex items-center gap-3">
                  <span className="text-gray-500">{it.qty} ×</span>
                  <strong className="whitespace-nowrap text-h5 lg:text-h4">
                    {rials(it.price)}
                  </strong>
                </div>
              </li>
            ))}
          </ul>

          {/* Expand for mobile */}
          {hiddenCount > 0 && (
            <div className="mt-2 flex justify-center sm:hidden">
              <button
                type="button"
                onClick={() => setExpanded((s) => !s)}
                className="text-sm text-pink-600 hover:text-pink-700 inline-flex items-center gap-2 font-medium"
              >
                <span>{expanded ? "بستن" : `موارد بیشتر`}</span>
                {!expanded && (
                  <span className="text-gray-400">(+{hiddenCount})</span>
                )}
              </button>
            </div>
          )}
        </section>
        {/* Summary (left in desktop) */}
        <aside className="rounded-2xl border-t border-gray-200 bg-white p-6 lg:col-span-4">
          <dl className="text-sm space-y-3">
            <div className="flex items-center justify-between border-b border-dashed border-gray-200 pb-3">
              <dt className="text-gray-600">مجموع</dt>
              <dd className="font-bold">{rials(subtotal)}</dd>
            </div>
            <div className="flex items-center justify-between border-b border-dashed border-gray-200 pb-3">
              <dt className="text-gray-600">هزینه ارسال</dt>
              <dd className="font-bold">
                {delivery === 0 ? "رایگان" : rials(delivery)}
              </dd>
            </div>
            <div className="text-base flex items-center justify-between pt-1">
              <dt className="font-bold">جمع کل</dt>
              <dd className="font-extrabold">{rials(total)}</dd>
            </div>
          </dl>
        </aside>
      </section>
    </main>
  );
}
