import React from "react";
import { BaseFood, FoodDetail, Variety } from "@/src/types/food";
import { useChefAvailableQuery } from "@/src/hooks/Queries";
import { useCartStore } from "@/src/store";
import NextImg from "@/src/components/NextImage";
import { numberWithCommas } from "@/src/utils/Helpers";
import AddCardButton from "@/src/components/AddCardButton";
import { useCart } from "@/src/hooks";
import Link from "next/link";
import { useSnackbar } from "notistack";
import { ProductTumbSlider } from "../ProductTumbSlider";

interface Props {
  productData: FoodDetail;
}
function ProductDetail({ productData }: Props) {
  const { data } = useChefAvailableQuery(productData.chef.id);
  const { itemCount } = useCartStore();
  const { handleAddCard, handleRemoveCard, isLoading } = useCart();
  const { enqueueSnackbar } = useSnackbar();

  const AddCartHandler = async (product: BaseFood, variety: Variety) => {
    try {
      await handleAddCard(product, variety);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };
  const removeFromCartHandler = async (varietyID: string) => {
    try {
      await handleRemoveCard(varietyID);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };
  return (
    <div className="flex w-full flex-col gap-12 p-0 lg:flex-row lg:p-1">
      <div className="w-full max-w-[450px]">
        <ProductTumbSlider
          gallery={productData.gallery}
          image={productData.image}
        />
      </div>
      <div className=" flex w-full  ">
        <div className="mb-3 flex w-full flex-col gap-12 ">
          <Link
            href={`/chef/${productData.chef.id}`}
            className=" flex  w-full items-center  justify-start gap-2"
          >
            <NextImg
              imgWrapperClassName="  size-16   rounded-base"
              src={productData.chef.logo ?? ""}
              width={32}
              height={32}
              alt={productData.chef.title}
            />
            <h2 className="  text-h4 ">{productData.chef.title}</h2>
          </Link>
          <h1 className="w-full   text-h3 font-bold ">{productData.title}</h1>
          {productData.varieties.map((item) => (
            <div
              key={item.varietyID}
              className="flex w-full items-center  justify-between border-b border-black/20 py-8"
            >
              <div>
                <span className="text-h3">{item.title}</span>
                <div className="flex items-center gap-2">
                  {+item.price !== 0 && (
                    <>
                      {item.discount?.percent !== 0 && (
                        <>
                          <div
                            className="flex h-10 min-w-10 items-center justify-center rounded-base bg-primary  text-center
                           text-h6  text-white"
                          >
                            {item.discount?.percent} %
                          </div>
                          <span className="text-h5  font-light text-secondary line-through ">
                            <span>{numberWithCommas(item.price)}</span>
                          </span>
                        </>
                      )}
                      <p className="text-h3 font-bold">
                        {numberWithCommas(item.discount?.discountedPrice ?? 0)}{" "}
                        <span className="text-h5 font-normal">تومان</span>
                      </p>
                    </>
                  )}
                </div>
              </div>
              {data && (
                <AddCardButton
                  isLoading={isLoading}
                  onAddCard={() => AddCartHandler(productData, item)}
                  onRemoveCard={() => removeFromCartHandler(item.varietyID)}
                  productInCart={itemCount(item.varietyID)}
                  limit={item.maxOrderLimit}
                  quantity={item.quantity}
                />
              )}
            </div>
          ))}
        </div>
      </div>
    </div>
  );
}

export default ProductDetail;
