import React, { useCallback, useRef } from "react";
import { Swiper, SwiperRef, SwiperSlide } from "swiper/react";
import { Navigation } from "swiper/modules";
import { ArrowLeft2, ArrowRight2 } from "iconsax-react";
import ProductCard from "@/src/components/ProductCard";
import { usePaginatedFoodsQuery } from "@/src/hooks/Queries";
import { OrderDataVariant } from "@/src/types/general";
import { BaseFood } from "@/src/types/food";
import { useSnackbar } from "notistack";
import { useCartStore } from "@/src/store";

interface Props {
  chefID: string;
}
function ProductSlider({ chefID }: Props) {
  const sliderRef = useRef<SwiperRef>(null);

  const handlePrev = useCallback(() => {
    if (!sliderRef.current) return;
    sliderRef.current.swiper.slidePrev();
  }, []);

  const handleNext = useCallback(() => {
    if (!sliderRef.current) return;
    sliderRef.current.swiper.slideNext();
  }, []);
  const { data } = usePaginatedFoodsQuery({
    order: OrderDataVariant.DESC,
    page: 1,
    chefID,
  });
  const { enqueueSnackbar } = useSnackbar();
  const { addToCard, itemCount, removeFromCard } = useCartStore();

  const handleAddCard = (productData: BaseFood) => {
    addToCard({
      id: productData.id,
      count: 1,
      price: productData.price,
      title: productData.title,
      varietyID: productData.varietyID,
      chefID: productData.chefID,
    });
    enqueueSnackbar("غذا با موفقیت به سبد خرید شما اضافه شد", {
      variant: "success",
    });
  };
  const handleRemoveFromCard = (varietyID: string) => {
    removeFromCard(varietyID);
    enqueueSnackbar("غذا با موفقیت از سبد خرید شما کم شد", {
      variant: "success",
    });
  };
  return (
    <div className="  w-full ">
      <div className="flex w-full items-center justify-between">
        <p className="mb-2 text-body">غذاهای دیگر این رستوران</p>
        <div className="flex items-center gap-4 ">
          <button
            className="flex h-5 w-3 items-center justify-center  lg:hidden "
            onClick={handlePrev}
          >
            <ArrowRight2 size="18" color="#ffffff" variant="Broken" />
          </button>{" "}
          <button
            className="flex h-5 w-3 items-center justify-center lg:hidden "
            onClick={handleNext}
          >
            <ArrowLeft2 size="18" color="#ffffff" variant="Broken" />
          </button>
        </div>
      </div>
      <div className="flex items-center">
        <button
          className="hidden h-5 w-3 items-center justify-center rounded-s-8  bg-secondary lg:flex "
          onClick={handlePrev}
        >
          <ArrowRight2 size="18" color="#ffffff" variant="Broken" />
        </button>
        <Swiper
          ref={sliderRef}
          navigation={true}
          modules={[Navigation]}
          breakpoints={{
            240: {
              slidesPerView: 1,
            },
            400: {
              slidesPerView: 1,
            },
            540: {
              slidesPerView: 2,
            },
            670: {
              slidesPerView: 2,
            },
            890: {
              slidesPerView: 2,
            },
            1280: {
              slidesPerView: 3,
            },
          }}
          spaceBetween={21}
          className=" w-full"
        >
          {data?.data.map((item) => (
            <SwiperSlide key={item.id}>
              <ProductCard
                discount={item.discount}
                countInCard={itemCount(item.varietyID)}
                onAddCard={() => handleAddCard(item)}
                onRemoveCard={() => handleRemoveFromCard(item.varietyID)}
                title={item.title}
                Summary={item.Summary}
                image={item.image}
                price={item.price}
                id={item.id}
                calory={item.calory}
                limit={item.maxOrderLimit}
              />
            </SwiperSlide>
          ))}
        </Swiper>

        <button
          onClick={handleNext}
          className=" hidden h-5 w-3 items-center justify-center rounded-e-8  bg-secondary lg:flex "
        >
          <ArrowLeft2 size="18" color="#ffffff" variant="Broken" />
        </button>
      </div>
    </div>
  );
}

export default ProductSlider;
