import React, { useEffect, useState } from 'react'
import FoodsSideBar from '../../Foods/FoodsSideBar';
import { useCategoriesQuery } from '@/src/hooks/Queries';
import NextImg from "@/src/components/NextImage";
import { ArrowDown2, ArrowUp, ArrowUp2 } from 'iconsax-react';
import Skeleton from 'react-loading-skeleton';
import { useSearchParams, usePathname, useRouter } from "next/navigation";


interface Props {
    setSelectedID: React.Dispatch<React.SetStateAction<string | undefined>>;
    selectedID?: string
}

const ProductCategory = ({
    setSelectedID,
    selectedID
}: Props) => {
    const [selectedParentID, setSelectedParentID] = useState<string | undefined>(undefined);
    const [hasCategory, setHasCategory] = useState(false);

    const searchParams = useSearchParams();
    const pathname = usePathname();
    const router = useRouter();

    const { data: categories } = useCategoriesQuery({});

    const { data: childCategories, isLoading } = useCategoriesQuery({ parentID: selectedParentID });

    const handleCategoryClick = (id: string) => {

        if (id !== selectedParentID) {
            setSelectedParentID(id);
        } else {
            setSelectedParentID(undefined);
        }
    };

    const handleCategorySelected = (id: string) => {
        setSelectedID(id)
    }

    useEffect(() => {
        setHasCategory(searchParams.has("category"));
    }, [searchParams]);



    const removeSelectedCategory = () => {
        const params = new URLSearchParams(searchParams.toString()); // Clone existing params
        params.delete("category"); // Remove category param

        router.replace(`${pathname}?${params.toString()}`);
    };

    return (
        <div className="flex h-fit mt-7 w-full rounded-half border-black/20 p-8 lg:justify-between lg:border">
            <div className="flex w-full flex-col items-start max-h-[590px] md:max-h-[630px] justify-start overflow-auto gap-8 px-12 text-h4">
                <div className='flex w-full justify-between'>
                    <p className="text-h3 font-bold">دسته‌بندی</p>
                    {(hasCategory) ? <div className='flex'>

                        <span onClick={removeSelectedCategory} style={{ cursor: 'pointer' }}
                            className="inline-flex items-center rounded-half bg-gray-50 px-2 py-1 text-xs font-medium text-gray-600 ring-1 ring-gray-500/10 ring-inset">
                            حذف فیلتر x
                        </span>

                    </div> : ''}
                </div>

                {categories?.map((item) => (
                    <button key={item.id}
                        className="flex flex-col w-full items-center justify-between py-2 text-h3 font-bold"
                    >
                        <div className="w-full flex justify-between">
                            <div className='flex relative w-3/4 items-center'>
                                <NextImg src={item.image as string} alt={item.title} width={32} height={32} />

                                <p onClick={() => handleCategorySelected(item.id)} className="text-h4 mr-4 font-bold">{item.title}</p>
                            </div>
                            {item.child_count > 0 ? (
                                <span className={`rounded-full p-[4px]`}>
                                    <ArrowUp2 onClick={() => handleCategoryClick(item.id)}
                                        size="16"
                                        variant="Broken"
                                        className={`duration-700 ${selectedParentID === item.id ? '' : 'rotate-180'} `}
                                    />
                                </span>
                            ) : ''}
                        </div>
                        <div style={{ alignSelf: 'self-start' }}>
                            <div className={`mr-15 mb-2 pr-4 w-full border-r-2 mt-2 ${selectedParentID === item.id ? '' : 'hidden'}`}>
                                {isLoading ?
                                    Array(6)
                                        .fill(0)
                                        .map((_, index) => (
                                            <div key={index} className="w-full">
                                                <Skeleton className="h-full w-full" />
                                            </div>
                                        )) : ''
                                }

                                {selectedParentID ? (
                                    childCategories?.map((child) => (
                                        <div onClick={() => handleCategorySelected(child.id)} className='flex mb-3 w-full relative'>
                                            <NextImg src={item.image as string} alt={item.title} width={16} height={16} />
                                            <p key={child.id} className="text-h5 py-1 text-right font-medium">
                                                {child.title}
                                            </p>
                                        </div>
                                    ))
                                ) : ''}

                            </div>
                        </div>
                    </button>

                ))}
            </div>
        </div>
    )
}

export default ProductCategory