import { TextField, FormProvider, DatePicker } from "@/src/components/RHF";
import React, { useEffect, useMemo } from "react";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as Yup from "yup";
import { useAuth } from "@/src/hooks";
import Button from "@/src/components/Button";
import { UserSquare } from "iconsax-react";
import { useAuthStore } from "@/src/store";
import { BirthDay } from "@/src/types/auth";
import { useSnackbar } from "notistack";

interface FormFields {
  name: string;
  family: string;
  email?: string;
  phone?: string;
  birthDay?: BirthDay;
}

function Info() {
  const { updateUser } = useAuth();
  const { authUser } = useAuthStore();

  const { enqueueSnackbar } = useSnackbar();
  const defaultValues: FormFields = useMemo(
    () => ({
      name: authUser?.name ?? "",
      family: authUser?.family ?? "",
      email: authUser?.email ?? "",
      phone: authUser?.phone ?? "",
      birthDay: authUser.birthDay ?? {
        day: 0,
        month: 0,
        year: 0,
      },
    }),
    [authUser]
  );

  const registerSchema = useMemo(
    () =>
      Yup.object().shape({
        name: Yup.string().required("لطفا نام خود را وارد نمایید "),
        family: Yup.string().required("لطفا نام خانوادگی خود را وارد نمایید "),
        email: Yup.string()
          .email("لطفا آدرس ایمیل خود را به درستی وارد کنید ")
          .optional(),
        phone: Yup.string()
          .test(
            "mobileValid",
            "لطفا شماره تلفن خود را بصورت درست وارد کنید ",
            (value) => {
              if (value) {
                return /^0[1-8]{2,}[0-9]{8,}$/.test(value);
              }
              return true;
            }
          )
          .optional(),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );
  const methods = useForm<FormFields>({
    defaultValues,
    resolver: yupResolver(registerSchema),
    mode: "onChange",
  });
  const {
    handleSubmit,
    formState: { isValid, isSubmitting },
    reset,
  } = methods;
  useEffect(() => {
    reset({
      name: authUser?.name ?? "",
      family: authUser?.family ?? "",
      email: authUser?.email ?? "",
      phone: authUser?.phone ?? "",
      birthDay: authUser.birthDay ?? {
        day: 0,
        month: 0,
        year: 0,
      },
    });
  }, [authUser]);

  const onSubmit = handleSubmit(async (data) => {
    try {
      await updateUser(data);
      enqueueSnackbar("تغییرات با موفقیت انجام شد", {
        variant: "success",
      });
    } catch (error) {
      console.log(error);
      enqueueSnackbar("خطا در ثبت اطلاعات", {
        variant: "error",
      });
    }
  });
  return (
    <div className="flex flex-col ">
      <div className="flex w-full gap-4 ">
        <UserSquare />
        <p className="text-h3  ">اطلاعات کاربری</p>
      </div>
      <FormProvider methods={methods} onSubmit={onSubmit}>
        <div className="grid grid-cols-1 gap-16 px-4  py-16 lg:px-28 2xl:grid-cols-2">
          <TextField fullWidth type="text" name="name" label="نام*" />{" "}
          <TextField
            fullWidth
            type="text"
            name="family"
            label="نام خانوادگی*"
          />{" "}
          <TextField
            fullWidth
            type="text"
            name="phone"
            placeholder="برای مثال : 03537202020"
            label="تلفن ثابت"
            inputMode="decimal"
          />{" "}
          <DatePicker fullwidth name="birthDay" title="تاریخ تولد" />
          <TextField
            fullWidth
            type="email"
            name="email"
            label="ایمیل"
            placeholder=" mihmannavaz@gmail.com"
          />
          <span></span>
          <Button
            fullWidth
            type="submit"
            isLoading={isSubmitting}
            disabled={!isValid}
          >
            ثبت تغییرات
          </Button>
        </div>
      </FormProvider>
    </div>
  );
}

export default Info;
