import { FormProvider, TextArea } from "@/src/components/RHF";
import React, { useMemo } from "react";
import { useFieldArray, useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as Yup from "yup";
import Button from "@/src/components/Button";

import { useSnackbar } from "notistack";
import ReactStars from "react-rating-stars-component";
import { useComments } from "@/src/hooks";
import { Order } from "@/src/types/Cart";
import NextImg from "@/src/components/NextImage";

import { CommentFormFields } from "@/src/types/Comment";

interface Props {
  onVerify: () => void;
  order: Order;
}

function CommentForm({ onVerify, order }: Props) {
  const commentSchema = useMemo(
    () =>
      Yup.object().shape({
        content: Yup.string().required("متن نظر اجباری است ").defined(),
        cartItems: Yup.array()
          .of(
            Yup.object()
              .shape({
                rate: Yup.number()
                  .required("لطفا نظر خود را ثبت کنید ")
                  .min(1, "لطفا نظر خود را ثبت کنید "),
                id: Yup.string().defined().required(),
              })
              .defined()
              .required()
          )
          .defined(),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );
  const defaultValues: CommentFormFields = useMemo(
    () => ({
      content: "",
      cartItems: order.cart.map((item) => ({
        id: item.variety.varietyID,
        rate: 0,
      })),
    }),
    []
  );
  const methods = useForm<CommentFormFields>({
    defaultValues,
    resolver: yupResolver(commentSchema),

    mode: "onChange",
  });

  const {
    handleSubmit,
    formState: { isValid, isSubmitting },

    control,
  } = methods;

  const { enqueueSnackbar } = useSnackbar();
  const { update } = useFieldArray<CommentFormFields>({
    control,
    name: "cartItems",
  });
  const { sendComment } = useComments();
  const onSubmit = handleSubmit(async (values) => {
    try {
      await sendComment(order.id, values);

      enqueueSnackbar("نظر شما با موفقیت ارسال شد", {
        variant: "success",
      });
      onVerify();
    } catch (error) {
      console.log(error);
      enqueueSnackbar(error?.message ?? "خطا در ارسال نظر", {
        variant: "error",
      });
    }
  });

  return (
    <div className="flex  flex-col  lg:px-16">
      <FormProvider methods={methods} onSubmit={onSubmit}>
        <div className=" flex w-full flex-col gap-12 ">
          {order.cart.map((item, index) => (
            <div
              key={item.variety.varietyID}
              className="flex items-center  justify-between "
            >
              <div className="text flex gap-4 text-h3">
                <div className="size-40">
                  <NextImg
                    src={item.variety.image ?? ""}
                    alt=""
                    width={67}
                    height={58}
                  />
                </div>
                <div>
                  <p>{item.food.title}</p>
                  <p className="text-h5">{item.variety.title}</p>
                </div>
              </div>
              <ReactStars
                count={5}
                onChange={(value) => {
                  update(index, {
                    ...item,
                    rate: value,
                    id: item.variety.varietyID,
                  });
                }}
                size={24}
                isHalf={false}
                activeColor="#ffd700"
              />
            </div>
          ))}
          <hr />
          <TextArea
            name="content"
            label=""
            rows={10}
            placeholder="نظر خود را در مورد غذا وارد کنید"
            className="w-full"
          />
          <Button type="submit" isLoading={isSubmitting} disabled={!isValid}>
            ثبت نظر
          </Button>
        </div>
      </FormProvider>
    </div>
  );
}

export { CommentForm };
