import {
  TextField,
  FormProvider,
  TextArea,
  Select,
} from "@/src/components/RHF";
import React, { useMemo } from "react";
import { useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as Yup from "yup";
import Button from "@/src/components/Button";

import { useSnackbar } from "notistack";
import { useTicket } from "@/src/hooks";
import { FormOption } from "@/src/types/general";
import { useTicketCategoriesQuery } from "@/src/hooks/Queries";
import FileUploader from "@/src/components/FileUploader";
import { TicketItem } from "@/src/types/ticket";

interface FormFields {
  title: string;
  description: string;
  category: FormOption;
  file?: string;
}

interface Props {
  onAddTicket: VoidFunction;
  parentData?: TicketItem;
}
function TicketForm({ onAddTicket, parentData }: Props) {
  const ticketSchema = useMemo(
    () =>
      Yup.object().shape({
        title: Yup.string().required("لطفا موضوع تیکت را وارد نمایید "),
        description: Yup.string().required(
          "لطفا متن درخواست خود را وارد نمایید "
        ),
        category: Yup.object().shape({
          label: Yup.string().required(
            "لطفا دسته بندی مدنظر خود را وارد نمایید "
          ),
          value: Yup.string().required(
            "لطفا دسته بندی مدنظر خود را وارد نمایید "
          ),
        }),
      }),
    // eslint-disable-next-line react-hooks/exhaustive-deps
    []
  );

  const defaultValues: FormFields = useMemo(
    () => ({
      description: "",
      title: parentData?.title || "",
      category: {
        label: parentData?.categoryTitle || "",
        value: parentData?.categoryId || "",
      },
    }),
    []
  );
  const methods = useForm<FormFields>({
    defaultValues,
    resolver: yupResolver(ticketSchema),
    mode: "onChange",
  });

  const {
    handleSubmit,
    formState: { isValid, isSubmitting },
    setValue,
    reset,
  } = methods;
  const { createTicket } = useTicket();
  const { enqueueSnackbar } = useSnackbar();

  const onSubmit = handleSubmit(async (data) => {
    try {
      await createTicket(
        {
          ...data,
          categoryId: data.category.value,
        },
        parentData?.id ?? ""
      );
      enqueueSnackbar("تیکت با موفقیت ارسال شد", {
        variant: "success",
      });
      reset();
      onAddTicket();
    } catch (error) {
      console.log(error);
      enqueueSnackbar("خطا در ثبت اطلاعات", {
        variant: "error",
      });
    }
  });
  const { data } = useTicketCategoriesQuery();

  return (
    <>
      <div className="flex flex-col  px-16 py-12">
        <FormProvider methods={methods} onSubmit={onSubmit}>
          <div className=" flex w-full flex-col gap-8">
            {!parentData && (
              <>
                <div className="grid grid-cols-1   ">
                  <TextField
                    WrapperClassName="max-w-none"
                    className="w-full "
                    type="text"
                    name="title"
                    label="موضوع *"
                  />{" "}
                </div>
                <div className="grid w-full grid-cols-1  ">
                  <Select
                    placeholder="دسته بندی"
                    label="دسته بندی"
                    options={data}
                    name="category"
                    fullWidth
                  />
                </div>
              </>
            )}
            <div className="flex flex-col gap-4">
              <TextArea
                fullWidth
                rows={5}
                name="description"
                label="متن تیکت"
              ></TextArea>
            </div>
            <div className="flex flex-col gap-4">
              {!isSubmitting && (
                <FileUploader
                  onUploadComplete={(path) => setValue("file", path)}
                />
              )}
            </div>

            <Button
              fullWidth
              type="submit"
              isLoading={isSubmitting}
              disabled={!isValid}
            >
              ارسال تیکت
            </Button>
          </div>
        </FormProvider>
      </div>
    </>
  );
}

export default TicketForm;
