import { FormProvider, Radio, Select } from "@/src/components/RHF";
import React, { useMemo } from "react";
import { useForm } from "react-hook-form";
import Button from "@/src/components/Button";
import { useSnackbar } from "notistack";
import { ChefAvailable } from "@/src/types/Chef";
import { convertEnglishWeekdayToPersian } from "@/src/utils/Helpers";
import { useChefPerOrderTimesQuery } from "@/src/hooks/Queries";
import { FormOption } from "@/src/types/general";
import { parseAsString, useQueryStates } from "next-usequerystate";
import { useCartStore } from "@/src/store";

interface FormFields {
  day?: FormOption;
  time?: FormOption;
  type?: string;
}

interface Props {
  onSelect: () => void;
  nearestOpeningTime: ChefAvailable;
  chefId: string;
}
function PreOrderTimeForm({ onSelect, nearestOpeningTime, chefId }: Props) {
  // eslint-disable-next-line no-unused-vars, @typescript-eslint/no-unused-vars
  const [queryFilters, setQueryFilters] = useQueryStates({
    time: parseAsString.withDefault(""),
    day: parseAsString.withDefault(""),
  });
  const defaultValues: FormFields = useMemo(
    () => ({
      day: undefined,
      time: undefined,
      type: "1",
    }),
    []
  );

  const methods = useForm<FormFields>({
    defaultValues,
    // resolver: yupResolver(addressSchema),
    mode: "onChange",
  });

  const {
    handleSubmit,
    formState: { isValid, isSubmitting },
    reset,
    watch,
  } = methods;
  const day = watch("day");
  const type = watch("type");

  const { enqueueSnackbar } = useSnackbar();
  const { setDeliveryTime } = useCartStore();

  const onSubmit = handleSubmit(async (values) => {
    try {
      if (type === "2") {
        setQueryFilters({
          day: values.day?.value,
          time: values.time?.value,
        });
        setDeliveryTime(values.day?.value ?? "", values.time?.value ?? "");
      } else {
        setQueryFilters({
          day: nearestOpeningTime.title,
          time: nearestOpeningTime.time[0].from,
        });
        setDeliveryTime(
          nearestOpeningTime.title,
          nearestOpeningTime.time[0].from
        );
      }
      reset();
      onSelect();
    } catch (error) {
      console.log(error);
      enqueueSnackbar(" خطا در اتصال به درگاه پرداخت", {
        variant: "error",
      });
    }
  });
  const { data } = useChefPerOrderTimesQuery(chefId);
  const times = useMemo(
    () =>
      data?.times[day?.value ?? ""]?.map((item) => ({
        value: item,
        label: item,
      })) ?? [],
    [day, data]
  );
  return (
    <div className="flex  flex-col  ">
      <FormProvider methods={methods} onSubmit={onSubmit}>
        <div className=" flex w-full flex-col gap-12 ">
          <Radio
            options={[
              {
                label: "سریع ترین زمان ممکن",
                value: "1",
                description: `${convertEnglishWeekdayToPersian(nearestOpeningTime.title)} از ${nearestOpeningTime.time[0].from} تا ${nearestOpeningTime.time[0].to}`,
              },
              { label: "زمان دیگر", value: "2" },
            ]}
            name="type"
          />
          {type === "2" && (
            <div className=" flex items-center justify-center gap-12">
              <Select
                options={data?.days}
                name="day"
                label=""
                placeholder="روز"
              />
              <Select options={times} name="time" label="" placeholder="ساعت" />
            </div>
          )}

          <Button type="submit" isLoading={isSubmitting} disabled={!isValid}>
            تایید
          </Button>
        </div>
      </FormProvider>
    </div>
  );
}

export default PreOrderTimeForm;
