import ProductCard from "@/src/components/ProductCard";
import React, { useState } from "react";
import { CategoriesFood, FoodMenu, Variety } from "@/src/types/food";
import FoodsLoader from "@/src/containers/Foods/FoodsLoader";
import NextImg from "@/src/components/NextImage";
import { Modal } from "@/src/components/Modal";
import ProductPopupContent from "@/src/components/ProductPopupContent";
import { useCart, useDeleteActiveOrder } from "@/src/hooks";
import { useSnackbar } from "notistack";
import LoginModal from "@/src/components/LoginModal";
import { useAuthStore } from "@/src/store";
import ActiveOrderModal from "@/src/components/ActiveOrderModal";

interface Props {
  foods?: CategoriesFood[];
  discountFoods: FoodMenu[];
  available: boolean;

  isLoading: boolean;
}
function SingleRestaurantMenuItems({
  isLoading,
  available,
  foods,
  discountFoods,
}: Props) {
  const {
    handleAddCard,
    handleRemoveCard,
    isLoading: isLoadingCart,
  } = useCart();

  const { authUser } = useAuthStore();
  const { deleteActiveOrder, isLoading: isDeleteLoading } = useDeleteActiveOrder();

  const [targetFood, setTargetFood] = useState<FoodMenu | undefined>(undefined);
  const { enqueueSnackbar } = useSnackbar();
  const [isLoginModalOpen, setLoginModalOpen] = useState(false);
  const [isActiveOrderModalOpen, setActiveOrderModalOpen] = useState(false);
  const [pendingCartAction, setPendingCartAction] = useState<{
    product: FoodMenu;
    variety: Variety;
  } | null>(null);
  const AddCartHandler = async (product: FoodMenu, variety: Variety ) => {
    try {
      await handleAddCard(product, variety);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
      console.log('is here');
      
      // Check if error has order_active flag
      if (error?.data?.order_active === true) {
        console.log(error?.data?.order_active,'error?.data?.order_active');
        
        setPendingCartAction({ product, variety });
        setActiveOrderModalOpen(true);
        return;
      }
      
      if (error && !error.user_loged) {
        // setLoginModalOpen(true);
      }
    }
  };

  const handleDeleteActiveOrder = async () => {
    if (!authUser?.id || !pendingCartAction) return;
    
    try {
      const restaurantId = pendingCartAction.product.chefID;
      await deleteActiveOrder(Number(authUser.id), restaurantId);
      setActiveOrderModalOpen(false);
      
      // Retry the original cart action
      await handleAddCard(pendingCartAction.product, pendingCartAction.variety);
      setPendingCartAction(null);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در حذف سفارش فعال", {
        variant: "error",
      });
    }
  };

  const handleCloseActiveOrderModal = () => {
    setActiveOrderModalOpen(false);
    setPendingCartAction(null);
  };
  const removeFromCartHandler = async (varietyID: string) => {
    try {
      await handleRemoveCard(varietyID);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };

  return (
    <div className="flex flex-col gap-20 pb-20">
      {!isLoading && !!discountFoods.length && (
        <div id="menu_discount" className="flex  flex-col gap-20 ">
          <div className="flex w-auto max-w-max  items-center gap-4 whitespace-nowrap border-b border-primary">
            <p className="pr-1 text-h2 font-bold ">تخفیف دار ها</p>
          </div>
          <div className="grid grid-cols-1 gap-10">
            {discountFoods.map((product) => (
              <ProductCard
                onSelect={() => setTargetFood(product)}
                available={available}
                titleClassName=" xl:max-w-[150px]  2xl:max-w-[300px]"
                onAddCard={(variety) => AddCartHandler(product, variety)}
                isLoadingCart={isLoadingCart}
                onRemoveCard={(variety) => removeFromCartHandler(variety)}
                key={product.id}
                Summary={product.Summary}
                image={product.image}
                title={product.title}
                varieties={product.varieties}
              />
            ))}
          </div>
        </div>
      )}
      {isLoading && (
        <div className="grid grid-cols-1 gap-10">
          <FoodsLoader />
        </div>
      )}
      {foods?.map((item) => (
        <div key={item.id}>
          <span id={`menu_${item.id}`}></span>
          <div
            id={`item_${item.id}`}
            key={item.id}
            className="flex  flex-col gap-20 "
          >
            <div className="flex w-auto max-w-max  items-center gap-4 whitespace-nowrap border-b border-primary">
              <NextImg
                src={item.image || ""}
                imgWrapperClassName="size-18"
                alt={item.title}
                width={35}
                height={35}
              />
              <p className="pr-1 text-h2 font-bold ">{item.title}</p>
            </div>
            <div className="grid grid-cols-1 gap-10">
              {item.foods.map((product) => (
                <ProductCard
                  onAddCard={(variety) => AddCartHandler(product, variety)}
                  isLoadingCart={isLoadingCart}
                  onSelect={() => setTargetFood(product)}
                  available={available}
                  titleClassName=" lg:max-w-[90px] xl:max-w-[130px]  2xl:max-w-[300px]"
                  onRemoveCard={(variety) => removeFromCartHandler(variety)}
                  key={product.id}
                  Summary={product.Summary}
                  image={product.image}
                  title={product.title}
                  varieties={product.varieties}
                />
              ))}
            </div>
          </div>
        </div>
      ))}
      <Modal
        open={!!targetFood}
        title={targetFood?.title}
        wrapperClassName="max-w-2xl w-full"
        onClose={() => setTargetFood(undefined)}
      >
        {targetFood && (
          <ProductPopupContent
            targetFood={targetFood}
            available={available}
            onClose={() => setTargetFood(undefined)}
          />
        )}
      </Modal>

      <LoginModal
        open={isLoginModalOpen}
        onClose={() => setLoginModalOpen(false)}
      />

      <ActiveOrderModal
        open={isActiveOrderModalOpen}
        onClose={handleCloseActiveOrderModal}
        onConfirm={handleDeleteActiveOrder}
        isLoading={isDeleteLoading}
      />
    </div>
  );
}

export default SingleRestaurantMenuItems;
