import { parseAsString, useQueryState, useQueryStates } from "next-usequerystate";
import React, { useMemo, useState } from "react";
import { ChefDetail } from "@/src/types/Chef";
import { useTranslation } from "react-i18next";
import SingleRestaurantMenu from "./SingleRestaurantMenu";
import SingleRestaurantInfo from "./SingleRestaurantInfo";
import SingleRestaurantOrder from "../SingleRestaurantOrder";
import CategorySlider from "./SingleRestaurantMenu/CategorySlider";
import { useCategoriesFoodsQuery } from "@/src/hooks/Queries";
import { useScrollspy } from "@/src/hooks";
import {
  DeliveryDetails,
  DeliveryMethod,
} from "@/src/hooks/Queries/useDeliveryMethodByTime";
import { ServiceDetails } from "@/src/services/httpServices/CartServices/types";
import Image from "next/image";
import { Star1 } from "iconsax-react";

interface Props {
  chefData: ChefDetail;
  available: boolean;
  areaAddressData: ServiceDetails | undefined;
  setAvailableModal: React.Dispatch<React.SetStateAction<boolean>>;
  deliveryMethod: DeliveryDetails | undefined;
  deliveryOptions: DeliveryMethod[];
}

function makeFallbackAvatar(title: string) {
  const initial = (title?.trim()?.[0] || "🍽").toUpperCase();
  // deterministic pastel bg from title
  const hue = Array.from(title).reduce((a, c) => a + c.charCodeAt(0), 0) % 360;
  const bg = `hsl(${hue} 70% 90%)`;
  const fg = `hsl(${hue} 45% 35%)`;
  const svg = `
  <svg xmlns='http://www.w3.org/2000/svg' width='128' height='128'>
    <rect width='100%' height='100%' rx='999' fill='${bg}'/>
    <text x='50%' y='55%' dominant-baseline='middle' text-anchor='middle'
      font-family='system-ui, sans-serif' font-size='64' fill='${fg}' font-weight='700'>
      ${initial}
    </text>
  </svg>`;
  return `data:image/svg+xml;utf8,${encodeURIComponent(svg)}`;
}

function SingleRestaurantTabs({
  chefData,
  available,
  setAvailableModal,
  areaAddressData,
  deliveryMethod,
  deliveryOptions,
}: Props) {
  const [activeTab, setActiveTab] = useQueryState(
    "activeTab",
    parseAsString.withDefault("menu")
  );

  // Keep the product menu (category slider) in sync with the same query params
  // used inside the menu page.
  const [queryFilters] = useQueryStates({
    time: parseAsString.withDefault(""),
    day: parseAsString.withDefault(""),
    searchText: parseAsString.withDefault(""),
  });

  const fallback = useMemo(
    () => makeFallbackAvatar(chefData.title),
    [chefData.title]
  );
  const [src, setSrc] = useState(chefData.logo || fallback);
  console.log("🚀 ~ SingleRestaurantTabs ~ chefData:", chefData);

  // Fetch categories to render the menu header (full-width) above the split layout
  const { data: categories } = useCategoriesFoodsQuery({
    chefID: chefData.id,
    title: queryFilters.searchText,
    time: queryFilters.time,
    day: queryFilters.day,
  });

  const ids = useMemo(
    () => categories?.map((item) => `item_${item.id}`) ?? [],
    [categories]
  );
  const activeId = useScrollspy(ids, 400);

  const { t } = useTranslation();

  const apiActive =
    typeof areaAddressData?.active === "boolean"
      ? areaAddressData.active
      : undefined;

  const apiOpenBranch =
    typeof areaAddressData?.open_branch === "boolean"
      ? areaAddressData.open_branch
      : undefined;

  const apiAddressValid =
    typeof areaAddressData?.address_valid === "boolean"
      ? areaAddressData.address_valid
      : undefined;

  // optional: message coming from areaAddressData
  const apiMessage =
    typeof areaAddressData?.message === "string" &&
    areaAddressData.message.trim()
      ? areaAddressData.message
      : null;

  // optional: pre-order earliest time if provided by API (adjust field name if different)

  // compute the status result object
  const status = useMemo(() => {
    // prefer API booleans when available; else fall back to props
    const isActive = apiActive ?? available;
    // note: openBranch meaning can be "branch is accepting orders (open)" or "accepting preorders" depending on API semantics.
    const isOpenBranch = apiOpenBranch ?? chefData.hasPreOrder ?? false;
    const isAddressValid = apiAddressValid ?? true; // assume valid if unknown

    // Case 1: active + open + address invalid
    if (isActive && isOpenBranch && !isAddressValid) {
      return {
        text: "Restaurant is available but your address is outside the delivery area. Please change your address.",
        dot: "bg-warning-main",
        variant: "address-out-of-range",
      };
    }

    // Case 2: active + open + address valid
    if (isActive && isOpenBranch && isAddressValid) {
      return {
        text: "Restaurant is open. Please select a delivery time.",
        dot: "bg-Success-main",
        variant: "open",
      };
    }

    // Case 3: completely closed (not active, branch closed)
    if (!isActive && !isOpenBranch) {
      return {
        text: "Restaurant is currently closed.",
        dot: "bg-gray-500",
        variant: "closed",
      };
    }

    // Case 4: not active, open_branch true (accepting pre-orders) — address invalid
    if (!isActive && isOpenBranch && !isAddressValid) {
      return {
        text: "Restaurant is closed but accepting pre-orders. Your address is outside delivery area — please change address or choose earliest pre-order time.",
        dot: "bg-warning-main",
        variant: "preorder-address-invalid",
      };
    }

    // Other combinations: closed but accepting preorders and address valid, etc.
    if (!isActive && isOpenBranch && isAddressValid) {
      return {
        text: "Restaurant is closed but accepting pre-orders. Please select the earliest pre-order time.",
        dot: "bg-warning-main",
        variant: "preorder",
      };
    }

    // fallback to existing props if nothing matched
    if (available) {
      return {
        text: "Restaurant is open. Please select a delivery time.",
        dot: "bg-Success-main",
        variant: "open-fallback",
      };
    }
    if (chefData.hasPreOrder) {
      return {
        text: "Accepting pre-orders.",
        dot: "bg-warning-main",
        variant: "preorder-fallback",
      };
    }

    return {
      text: "Restaurant is currently closed.",
      dot: "bg-gray-500",
      variant: "closed-fallback",
    };
  }, [
    apiActive,
    apiOpenBranch,
    apiAddressValid,
    available,
    chefData.hasPreOrder,
  ]);

  // expose convenient values for JSX
  const dotClass = status.dot;

  const rating = Number(chefData?.rate ?? 0);

  const showMenu = activeTab === "menu";

  return (
    <div className="" dir="rtl">
      {/* Header row (full width) */}
      <div className="container flex items-center justify-between gap-3 border-b border-black/10 px-4 py-4 lg:px-0">
        <div className="flex min-w-0 items-center gap-3">
          {/* Logo */}
          <div className="relative h-20 w-20 shrink-0 overflow-hidden rounded-full ring-1 ring-black/5 md:h-24 md:w-24">
            <Image
              src={src}
              alt={chefData.title || "لوگو"}
              fill
              sizes="96px"
              className="object-cover"
              onError={() => setSrc(fallback)}
              priority
            />
          </div>

          {/* Title + Status */}
          <div className="min-w-0">
            <div className="flex items-center gap-3">
              <h1 className="truncate text-h2 font-bold text-gray-900">
                {chefData.title}
              </h1>
              {rating > 0 && (
                <div className="flex items-center gap-1.5 text-h5 text-gray-800">
                  <Star1 size={16} color="#FFC33F" variant="Bold" />
                  <span>{rating}</span>
                </div>
              )}
            </div>

            <div className="mt-1 flex items-center gap-2 text-gray-800">
              <span className={`h-3 w-3 rounded-full ${dotClass}`} />
              <span className="truncate text-h5 text-gray-800">
                {areaAddressData?.message || ""}
              </span>
            </div>
          </div>
        </div>

        {/* Switch (menu / info) next to title row */}
        <div className="shrink-0">
          <div className="flex items-center rounded-[6px] bg-[rgb(255_253_116/0.35)] p-1 backdrop-blur">
            <button
              type="button"
              onClick={() => setActiveTab("menu")}
              className={`whitespace-nowrap rounded-[6px] px-3 py-2 text-h5 transition-all md:px-4 ${
                showMenu
                  ? "bg-[rgb(255_253_116/1)] font-bold text-[#0a0aa8] shadow-sm"
                  : "text-[#0a0aa8]/80 hover:bg-[rgb(255_253_116/0.55)]"
              }`}
            >
              منو
            </button>
            <button
              type="button"
              onClick={() => setActiveTab("info")}
              className={`whitespace-nowrap rounded-[6px] px-3 py-2 text-h5 transition-all md:px-4 ${
                !showMenu
                  ? "bg-[rgb(255_253_116/1)] font-bold text-[#0a0aa8] shadow-sm"
                  : "text-[#0a0aa8]/80 hover:bg-[rgb(255_253_116/0.55)]"
              }`}
            >
              اطلاعات مجموعه
            </button>
          </div>
        </div>
      </div>

      {/* Content */}
      <div className="container px-4 pb-10 pt-4 lg:px-0">
        {showMenu ? (
          <>
            {/* Product menu (full width) */}
            {Array.isArray(categories) && categories.length > 0 && (
              <CategorySlider
                activeId={activeId}
                data={categories}
                wrapperClassName="sticky top-0 z-40 mb-6 rounded-[6px] bg-[rgb(255_253_116/0.35)] p-1 backdrop-blur"
              />
            )}

            {/* After menu: split layout */}
            <div className="grid grid-cols-1 gap-6 md:grid-cols-12">
              {/* Products (right) */}
              <div className="md:col-span-9 md:order-1">
                <SingleRestaurantMenu
                  setAvailableModal={setAvailableModal}
                  deliveryMethod={deliveryMethod}
                  areaAddressData={areaAddressData}
                  deliveryOptions={deliveryOptions}
                  chefID={chefData.id}
                  available={available}
                  hideCategorySlider
                  disableContainer
                />
              </div>

              {/* Order sidebar (left) */}
              <div className="md:col-span-3 md:order-2">
                <SingleRestaurantOrder
                  areaAddressData={areaAddressData}
                  deliveryMethod={deliveryMethod}
                  setAvailableModal={setAvailableModal}
                  chefData={chefData}
                  deliveryOptions={deliveryOptions}
                />
              </div>
            </div>
          </>
        ) : (
          <SingleRestaurantInfo chefData={chefData} />
        )}
      </div>
    </div>
  );
}

export default SingleRestaurantTabs;
