import {
  ChargeUserWalletService,
  checkCodeService,
  getUserProfileService,
  getUserWalletService,
  sendCodeService,
  updateProfileService,
  updateSelectedAddress,
} from "@/src/services/httpServices/AuthServices";
import { useState } from "react";
import Cookies from "js-cookie";
import { useAuthStore } from "@/src/store";
import { AuthUser } from "../../types/auth";
import { enqueueSnackbar } from "notistack";

function useAuth() {
  const [isLoading, setIsLoading] = useState(false);
  const { setAuthUser, resetAuthUser } = useAuthStore();

  const setTokenToCookies = async (token: string) => {
    Cookies.set("token", token || "", {
      expires: 7,
    });
  };
  const removeTokenFromCookies = async () => {
    Cookies.remove("token");
    Cookies.remove("Guest-token");
  };
  const getTokenFromCookies = () => {
    const token = Cookies.get("token") || "";
    return token;
  };
  const sendActivationCode = async (mobile: string) => {
    try {
      setIsLoading(true);
      await sendCodeService({
        mobile,
      });
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getUser = async () => {
    try {
      setIsLoading(true);
      const result = await getUserProfileService();
      const [year, month, day] = result.birth_date?.split("/") ?? [];
      const modeledUser: AuthUser = {
        id: result.id,
        family: result.family ?? "",
        defaultAddress: result.default_address_id ?? "",
        defaultAddressItem: result.default_address,
        mobile: result.mobile,
        name: result.name ?? "",
        email: result.email ?? "",
        phone: result.tel ?? "",
        inviteCode: result.referral_code,
        wallet_balance: result.wallet_balance ?? 0,
        birthDay: {
          day: day ? +day : 0,
          month: month ? +month : 0,
          year: year ? +year : 0,
        },
      };

      setAuthUser(modeledUser);
      return modeledUser;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const checkActivationCode = async (mobile: string, code: string) => {
    try {
      setIsLoading(true);
      const result = await checkCodeService({
        mobile,
        code,
      });
      setTokenToCookies(result.token);
      getUser();
      const modeledUser = {
        family: result.user.family ?? "",
        mobile: result.user.mobile,
        name: result.user.name ?? "",
      };
      return modeledUser;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const updateUser = async (user: AuthUser) => {
    try {
      setIsLoading(true);
      await updateProfileService({
        name: user.name ?? "",
        family: user.family ?? "",

        ...(user.mobile && {
          mobile: user.mobile,
        }),
        ...(user.email && {
          email: user.email,
        }),
        ...(user.phone && {
          tel: user.phone,
        }),
        ...(user.birthDay?.day && {
          day: user.birthDay?.day,
        }),
        ...(user.birthDay?.month && {
          month: user.birthDay?.month,
        }),
        ...(user.birthDay?.year && {
          year: user.birthDay?.year,
        }),
      });

      await getUser();
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const changeSelectedAddress = async (address: { id: string }) => {
    try {
      setIsLoading(true);
      await updateSelectedAddress({
        id: Number(address.id),
      });

      enqueueSnackbar("آدرس با موفقیت انتخاب شد", {
        variant: "success",
        preventDuplicate: true,
      });

      await getUser();
    } catch (error) {
      console.log(error);
      enqueueSnackbar("خطایی پیش آمده لطفا دوباره تلاش کنید", {
        variant: "error",
        preventDuplicate: true,
      });

      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const logoutUser = async () => {
    try {
      setIsLoading(true);
      removeTokenFromCookies();
      resetAuthUser();
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const getUserWallet = async () => {
    try {
      setIsLoading(true);
      const data = await getUserWalletService();
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const chargeUserWallet = async (amount: number) => {
    try {
      setIsLoading(true);
      const data = await ChargeUserWalletService({
        amount,
      });
      return data.code_pay;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  return {
    isLoading,
    changeSelectedAddress,
    sendActivationCode,
    checkActivationCode,
    setTokenToCookies,
    getUser,
    getTokenFromCookies,
    updateUser,
    logoutUser,
    getUserWallet,
    chargeUserWallet,
  };
}

export default useAuth;
