import {
  AddToCartService,
  CheckDiscountService,
  getCartService,
  getDeliveryTypesService,
  getGatewaysService,
  getOrderAvailableTimeService,
  getOrderService,
  getUserOrdersService,
  RemoveFromCartService,
  getAreaAddressService,
  setDeliveryService,
  setOrderFinalService,
  updateCartService,
  getDeliveryMethodByTimeService,
} from "@/src/services/httpServices/CartServices";
import {
  ApiAddCart,
  ApiSetAddress,
  ApiSetAddressByLocation,
  ApiSetDelivery,
} from "@/src/services/httpServices/CartServices/types";
import { useCartStore } from "@/src/store";
import {
  CartItem,
  DeliveryTime,
  DeliveryType,
  Gateway,
  Order,
  Payment,
  Pricing,
} from "@/src/types/Cart";
import { AuthUser } from "@/src/types/auth";
import { BaseFood, Variety } from "@/src/types/food";
import { OrderItems } from "@/src/types/general";
import moment from "jalali-moment";
import { useSnackbar } from "notistack";
import { useState } from "react";
import Cookies from "js-cookie";
import { useSelectionStore } from "@/src/store/useSelectionStore";
import { useQueryClient } from "@tanstack/react-query";

function UseCart() {
  const [isLoading, setIsLoading] = useState(false);
  const { addToCard, removeFromCard } = useCartStore();
  const queryClient = useQueryClient();
  const setGuestTokenToCookies = (token: string) => {
    if (token) {
      Cookies.set("Guest-token", token, {
        expires: 7,
      });
    }
  };

  const { enqueueSnackbar } = useSnackbar();
  const updateCartWitList = async (
    cartItems: CartItem[],
    deliveryTime?: DeliveryTime
  ) => {
    try {
      setIsLoading(true);
      const data: ApiAddCart[] = cartItems.map((item) => ({
        count: item.count,
        extraIds: "",
        variety_id: item.variety.varietyID,
        address_id: null,
        delivery_type_id: null,
      }));
      await updateCartService({
        carts: data,
        ...(deliveryTime && {
          time: deliveryTime.time,
          week_day: deliveryTime.day,
        }),
      });
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const getCart = async (params?: {
    delivery_type_id?: string;
    address_id?: string;
    branch_id?: string;
  }) => {
    try {
      setIsLoading(true);

      const result = await getCartService(params);

      const data: Order = {
        id: result.order.id,
        cart: result.order.order_cart.map((item) => ({
          id: item.id,
          food: {
            title: item.variety.product.title,
            id: item.variety.product.id,
            chefID: result.order.branch_id,
          },
          variety: {
            varietyID: item.variety.id,
            price: +item.variety.price,
            discount: {
              discountedPrice: +item.variety.discounted_price,
              percent: +item.variety.discount,
            },
            title: item.variety.title,
            maxOrderLimit: item.variety.max_order,
          },

          count: item.count,
          synced: true,
        })),
        pricing: {
          minOrder: result.order.branch?.min_order ?? 0,
          PackPrice: result.price.sumPaking ?? 0,
          sendPrice: result.price.priceShippingOrder ?? 0,
          tax: result?.price?.sumTax ? +result.price.sumTax : 0,
          total: result.price.sumProduct ?? 0,
          orderTotal: result.price.totalPayment ?? 0,
          discountedPrice: 0,
          cashBack: result.price.cashBack,
        },
        chef: {
          id: result.order.branch_id,
          name: result.order.branch.title,
        },
        address: {
          address: result.address?.address ?? "",
          id: result.address?.id ?? "",
          plaque: result.address?.plaque ?? "",
          postalCode: result.address?.postal_code ?? "",
          title: result.address?.title ?? "",
          unit: result.address?.unit ?? "",
          lat: result.address?.lat ?? "",
          lng: result.address?.lng ?? "",
        },
        deliveryType: result.order.delivery_type
          ? {
              descriptionErrorMsg:
                result.order.delivery_type?.description_error_text ?? "",
              descriptionRequired:
                result.order.delivery_type?.description_required ?? false,
              descriptionText:
                result.order.delivery_type?.description_text ?? "",
              isActive: result.order.delivery_type?.is_active ?? false,
              id: result.order.delivery_type?.id ?? "",
              isAddress: result.order.delivery_type?.is_address ?? false,
              isCost: result.order.delivery_type?.is_cost ?? false,
              title: result.order.delivery_type?.title ?? "",
              message: result.order.delivery_type?.message ?? "",
            }
          : undefined,
      };
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getUserOrders = async (type: OrderItems) => {
    try {
      setIsLoading(true);

      const result = await getUserOrdersService(type);

      const data: Order[] = result.map((item) => ({
        id: item.id,
        hasComment: item.comment_user !== null,
        cart:
          item.carts?.map((cart) => ({
            id: cart.id,
            food: {
              title: cart.variety?.product?.title ?? "",
              id: cart.variety.id,
              chefID: item.branch_id,
            },
            variety: {
              varietyID: cart.variety.id,
              price: +cart.variety.price,
              discount: {
                discountedPrice: cart.variety.discounted_price,
                percent: cart.variety.discount,
              },
              title: cart.variety.title,
              maxOrderLimit: cart.variety.max_order,
              image: cart.variety.product?.image ?? "",
            },

            count: cart.count,
            synced: true,
          })) ?? [],
        chef: {
          id: item.branch_id,
          name: item.branch.title,
        },
        pricing: {
          total: +(item.order_price ?? 0),
          discountedPrice: +(item.discount_price ?? 0),
          sendPrice: +(item.delivery_price ?? 0),
          orderTotal: +(item.order_price ?? 0),
          PackPrice: 0,
          tax: +(item.tax_price ?? 0),
          cashBack: 0,
        },
        address: {
          address: item.address ?? "",
          id: item.address_order?.id ?? "",
          plaque: item.address_order?.plaque ?? "",
          postalCode: item.address_order?.postal_code ?? "",
          title: item.address_order?.title ?? "",
          unit: item.address_order?.unit ?? "",
          lat: item.address_order?.lat ?? "",
          lng: item.address_order?.lng ?? "",
        },
        date: moment(item.created_at).locale("fa").format("D MMM yyyy"),
        status: item.order_status ?? undefined,
        gatewayCode: item.payment_code ?? "",
      }));
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const getUserOrder = async (id: string) => {
    try {
      setIsLoading(true);

      // call your service (assumes it returns the JSON you pasted)
      const resp: any = await getOrderService(id);

      // safe navigation to the order object
      const result = resp?.data?.order ?? resp?.order ?? resp;

      // some APIs return carts in different keys; prefer order_cart -> carts
      const cartsArray: any[] =
        result?.order_cart && Array.isArray(result.order_cart)
          ? result.order_cart
          : result?.carts && Array.isArray(result.carts)
            ? result.carts
            : [];

      // parse address string if needed
      let parsedAddress: any = {};
      if (result?.address) {
        try {
          parsedAddress =
            typeof result.address === "string"
              ? JSON.parse(result.address)
              : result.address;
        } catch (e) {
          // fallback to address_order object
          parsedAddress = result?.address_order ?? {};
        }
      } else {
        parsedAddress = result?.address_order ?? {};
      }

      // price block from top-level data.price if available
      const priceInfo = resp?.data?.price ?? {};

      const data: Order & { user: AuthUser; paymentCode: string } = {
        id: result?.id,
        status: result?.order_status,
        cart:
          cartsArray.map((cart: any) => {
            const variety = cart?.variety ?? {};
            const product = variety?.product ?? {};
            return {
              id: cart.id,
              food: {
                title: variety?.title ?? product?.title ?? cart?.title ?? "",
                id: variety?.id ?? cart?.variety_id ?? null,
                chefID: result?.branch_id ?? result?.branch?.id ?? null,
              },
              variety: {
                varietyID: variety?.id ?? cart?.variety_id ?? null,
                price: +(variety?.price ?? cart?.price ?? 0),
                discount: {
                  discountedPrice: +(
                    variety?.discounted_price ??
                    variety?.discounted_price ??
                    0
                  ),
                  percent: +(variety?.discount ?? cart?.discount ?? 0),
                },
                title: variety?.title ?? product?.title ?? "",
                maxOrderLimit: +(variety?.max_order ?? 0),
              },
              count: +(cart.count ?? 0),
              synced: true,
            };
          }) ?? [],
        chef: {
          id: result?.branch_id ?? result?.branch?.id ?? null,
          name: result?.branch?.title ?? result?.branch_name ?? "",
        },

        pricing: {
          // prefer the aggregated price object if present
          subtotal: +(priceInfo?.sumProduct ?? result?.order_price ?? 0),
          delivery: +(
            priceInfo?.deliveryPrice ??
            priceInfo?.priceShippingOrder ??
            result?.delivery_price ??
            0
          ),
          total:
            +(priceInfo?.sumTotal ?? result?.order_price ?? 0) +
            +(priceInfo?.deliveryPrice ?? result?.delivery_price ?? 0),
          minOrder: +(
            result?.branch?.min_order ??
            result?.branch?.minOrder ??
            0
          ),
          discountedPrice: +(
            result?.discount_price ??
            result?.discount_price ??
            0
          ),
          PackPrice: +(
            result?.packing_price ??
            result?.packing_price ??
            result?.packing_price ??
            0
          ),
          tax: +(priceInfo?.sumTax ?? result?.tax_price ?? 0),
          cashBack: +(priceInfo?.cashBack ?? result?.cash_back_price ?? 0),
        } as any,

        address: {
          address:
            parsedAddress?.address ??
            parsedAddress?.title ??
            result?.address_order?.address ??
            "",
          id:
            parsedAddress?.id ??
            result?.address_order?.id ??
            result?.address_id ??
            "",
          plaque: parsedAddress?.plaque ?? result?.address_order?.plaque ?? "",
          postalCode:
            parsedAddress?.postal_code ??
            parsedAddress?.postalCode ??
            result?.address_order?.postal_code ??
            null,
          title: parsedAddress?.title ?? parsedAddress?.address ?? "",
          unit: parsedAddress?.unit ?? result?.address_order?.unit ?? "",
          lat: parsedAddress?.lat ?? result?.address_order?.lat ?? "",
          lng: parsedAddress?.lng ?? result?.address_order?.lng ?? "",
        },

        date: moment(result?.created_at).locale("fa").format("D MMM yyyy"),
        user: {
          name: result?.user?.name ?? "",
          family: result?.user?.family ?? "",
          mobile: result?.user?.mobile ?? result?.user?.phone ?? "",
        },
        paymentCode:
          result?.payment_code ??
          result?.payment_code_gateway ??
          result?.paymentCode ??
          "",
      };

      return data;
    } catch (err) {
      console.log(err);
      throw err; // react-query will set isError, data=undefined
    } finally {
      setIsLoading(false);
    }
  };

  const setAddress = async (data: ApiSetAddress) => {
    try {
      setIsLoading(true);
      const result = await getAreaAddressService({
        branch_id: data.branch_id ? String(data.branch_id) : undefined,
        address_id: data.address_id ? String(data.address_id) : undefined,
        lat: data.lat ?? undefined,
        lng: data.lng ?? undefined,
        // delivery_type_id: data.delivery_type_id ? String(data.delivery_type_id) : undefined,
      });
      return result;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const setDelivery = async (deliveryData: ApiSetDelivery) => {
    try {
      setIsLoading(true);
      const result = await setDeliveryService({
        delivery_type_id: String(deliveryData.delivery_type_id),
        address_id: deliveryData.address_id || null,
      });
      const data: Pricing = {
        minOrder: result.order.branch.min_order,
        PackPrice: result.price.sumPaking ?? 0,
        sendPrice: result.price.priceShippingOrder ?? 0,
        tax: result?.price?.sumTax ? +result.price.sumTax : 0,
        total: result.price.sumProduct ?? 0,
        orderTotal: result.price.totalPayment ?? 0,
        discountedPrice: 0,
        cashBack: result.price.cashBack,
      };
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const CheckDiscount = async (code: string) => {
    try {
      setIsLoading(true);

      const result = await CheckDiscountService({
        coupon: code,
      });
      return result;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getGateway = async () => {
    try {
      setIsLoading(true);

      const result = await getGatewaysService();
      const data: Gateway[] = result.map((item) => ({
        id: +item.id,

        isActive: item.status,
        title: item.title,
      }));
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getDeliveryMethodByTime = async (data: ApiSetAddressByLocation) => {
    try {
      setIsLoading(true);
      const result = await getDeliveryMethodByTimeService({
        branch_id: data.branch_id,
        day: data.day,
        time: data.time,
      });
      return result;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
    }
  };

  const getDeliveryTypes = async (BranchId: string) => {
    try {
      setIsLoading(true);

      const result = await getDeliveryTypesService(BranchId);
      const data: DeliveryType[] = result.map((item) => ({
        descriptionErrorMsg: item.description_error_text,
        descriptionRequired: item.description_required,
        descriptionText: item.description_text,
        isActive: item.is_active,
        id: item.id,
        isAddress: item.is_address,
        isCost: item.is_cost,
        title: item.title,
        message: item.message ?? "",
      }));
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const getOrderAvailableTime = async (chefID: string, time: string) => {
    try {
      setIsLoading(true);
      const result = await getOrderAvailableTimeService(chefID, time);
      return result !== null;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const setOrderFinal = async (
    time: string,
    code?: string,
    description?: string
  ): Promise<Payment> => {
    try {
      setIsLoading(true);
      const result = await setOrderFinalService(time, code, description);
      return {
        code: result.payment_code ?? "",
      };
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const handleAddCard = async (product: BaseFood, variety: Variety) => {
    // Get selection from Zustand store using product.chefID
    const { getSelectionState } = useSelectionStore.getState();
    const selection = getSelectionState()[product.chefID];
    const selectedAddressId = selection?.selectedAddressId;
    const selectedDeliveryId = selection?.deliveryId;
    const selectedDeliveryTime = selection?.deliveryTime;

    try {
      setIsLoading(true);
      const result = await AddToCartService({
        variety_id: variety.varietyID,
        count: 1,
        week_day:
          selectedDeliveryTime?.day.toLowerCase() ??
          moment().format("dddd").toLowerCase(),
        time: selectedDeliveryTime?.time ?? moment().format("HH:mm:ss"),
        address_id: selectedAddressId ?? null, // Use Zustand value
        delivery_type_id: selectedDeliveryId ?? null, // Use Zustand value
      });

      // Set guest token immediately if it exists (synchronous)
      if (result.guest_token) {
        setGuestTokenToCookies(result.guest_token);
      }

      // Update store only after API success
      addToCard({
        count: 1,
        food: product,
        variety,
      });

      // Invalidate cart query to trigger refetch
      queryClient.invalidateQueries({ queryKey: ["cartList"] });

      enqueueSnackbar("محصول با موفقیت به سبد خرید شما اضافه شد", {
        variant: "success",
      });
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const handleRemoveCard = async (id: string) => {
    try {
      setIsLoading(true);
      await RemoveFromCartService({
        variety_id: id,
        count: 1,
      });
      
      removeFromCard(id);

      // Invalidate cart query to trigger refetch
      queryClient.invalidateQueries({ queryKey: ["cartList"] });

      enqueueSnackbar("محصول با موفقیت از سبد خرید شما کم شد", {
        variant: "success",
      });
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  return {
    isLoading,
    getDeliveryMethodByTime,
    updateCartWitList,
    getCart,
    setAddress,
    CheckDiscount,
    getGateway,
    getOrderAvailableTime,
    setOrderFinal,
    handleAddCard,
    handleRemoveCard,
    getUserOrders,
    getUserOrder,
    getDeliveryTypes,
    setDelivery,
  };
}

export default UseCart;
