import {
  ChefRegistrationService,
  getChefAvailableNowService,
  getChefAvailableTimeService,
  getChefPopupAlarm,
  getChefPreOrderTimeService,
  getChefService,
  getChefsService,
  getGalleryService,
} from "@/src/services/httpServices/ChefService";
import {
  BaseChef,
  ChefAvailableTimeList,
  ChefParams,
  ChefRegistration,
} from "@/src/types/Chef";
import {
  ApiFormOption,
  File,
  OrderDataVariant,
  PaginationResult,
} from "@/src/types/general";
import { convertEnglishWeekdayToPersian } from "@/src/utils/Helpers";
import { useState } from "react";

function UseChefs() {
  const [isLoading, setIsLoading] = useState(false);

  const getChefAvailableNow = async (chefID: string) => {
    try {
      setIsLoading(true);
      const result = await getChefAvailableNowService(chefID);
      return result !== null;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const ChefRegistration = async (data: ChefRegistration) => {
    try {
      setIsLoading(true);

      const birthday = data.birthDay
        ? `${data.birthDay.day}/${data.birthDay.month}/${data.birthDay.year}`
        : "";

      const socials = [
        { name: "instagram", link: data.instagram ?? "" },
        { name: "telegram", link: data.telegram ?? "" },
        { name: "youtube", link: data.youtube ?? "" },
      ];

      const formData = new FormData();
      formData.append("name", data.name);
      formData.append("family", data.family);
      formData.append("mobile", data.mobile);
      formData.append("address", data.address);
      formData.append("national_code", data.nationalCode);
      formData.append("tel", data?.phone ?? "");
      formData.append("birthday", birthday);
      formData.append("referral_code", data.referralCode ?? "");
      formData.append("user_name", data.username ?? "");

      // Handle file upload
      if (data.national_id_image?.[0]) {
        formData.append("national_id_image", data.national_id_image[0]);
      }

      if (data.business_license_image?.[0]) {
        formData.append(
          "business_license_image",
          data.business_license_image[0]
        );
      }

      socials.forEach((social, index) => {
        formData.append(`socials[${index}][name]`, social.name);
        formData.append(`socials[${index}][link]`, social.link);
      });

      const result = await ChefRegistrationService(formData);
      return result !== null;
    } catch (error) {
      console.error(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const getChefGallery = async (chefID: string) => {
    try {
      setIsLoading(true);
      const result = await getGalleryService(chefID);
      const meddledData: File[] =
        result[0]?.files.map((item) => ({
          id: item.id,
          path: item.path,
        })) ?? [];
      return meddledData;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getPopupAlarms = async (chefID: string) => {
    try {
      setIsLoading(true);
      const result = await getChefPopupAlarm({ branch_id: chefID });
      return result;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getPaginatedChefs = async ({
    order = OrderDataVariant.ASC,
    categoryID,
    page,
    title,
    lat,
    lng,
    paginate = 10,
  }: ChefParams) => {
    try {
      setIsLoading(true);
      const result = await getChefsService({
        order_by: order,
        ...(categoryID && {
          category_id: categoryID,
        }),
        ...(title && {
          title,
        }),
        ...(lat && {
          lat,
        }),
        ...(lng && {
          lng,
        }),
        per_page: paginate,

        page,
      });

      const modeledUser: PaginationResult<BaseChef> = {
        currentPage: result.current_page,
        perPage: result.per_page,
        total: result.total,
        totalPages: result.last_page,

        data: result.data.map((item) => ({
          id: item.id,
          title: item.title,
          logo: item.logo,
          cover: item.cover,
          ariaTitle: item?.area?.title ?? "--",
          rate: +item.rate,
          FoodsCounts: item.products_count,
          avgPrice: item?.products_with_variety_avg_price ?? 0,
        })),
      };

      return modeledUser;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getChefAvailableTime = async (chefID: string) => {
    try {
      setIsLoading(true);
      const result = await getChefAvailableTimeService(chefID);
      const data: ChefAvailableTimeList = {
        saturday: {
          title: result.saturday
            ? convertEnglishWeekdayToPersian(result.saturday[0].week_day)
            : "",
          time:
            result.saturday?.map((item) => ({
              from: item.from_time,
              to: item.to_time,
            })) ?? [],
        },
        sunday: {
          title: result.sunday
            ? convertEnglishWeekdayToPersian(result.sunday[0].week_day)
            : "",
          time:
            result.sunday?.map((item) => ({
              from: item.from_time,
              to: item.to_time,
            })) ?? [],
        },
        monday: {
          title: result.monday
            ? convertEnglishWeekdayToPersian(result.monday[0].week_day)
            : "",
          time:
            result.monday?.map((item) => ({
              from: item.from_time,
              to: item.to_time,
            })) ?? [],
        },
        tuesday: {
          title: result.tuesday
            ? convertEnglishWeekdayToPersian(result.tuesday[0].week_day)
            : "",
          time:
            result.tuesday?.map((item) => ({
              from: item.from_time,
              to: item.to_time,
            })) ?? [],
        },
        wednesday: {
          title: result.wednesday
            ? convertEnglishWeekdayToPersian(result.wednesday[0].week_day)
            : "",
          time:
            result.wednesday?.map((item) => ({
              from: item.from_time,
              to: item.to_time,
            })) ?? [],
        },
        thursday: {
          title: result.thursday
            ? convertEnglishWeekdayToPersian(result.thursday[0].week_day)
            : "",
          time:
            result.thursday?.map((item) => ({
              from: item.from_time,
              to: item.to_time,
            })) ?? [],
        },
        friday: {
          title: result.friday
            ? convertEnglishWeekdayToPersian(result.friday[0].week_day)
            : "",
          time:
            result.friday?.map((item) => ({
              from: item.from_time,
              to: item.to_time,
            })) ?? [],
        },
      };
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const getChefCenter = async (chefID: string) => {
    try {
      setIsLoading(true);
      const data = await getChefService(chefID); // your existing service
      return {
        id: String(data.id),
        title: data.title ?? "",
        lat: Number(data.lat) || 0,
        lng: Number(data.lng) || 0,
      };
    } finally {
      setIsLoading(false);
    }
  };

  const getChefPreOrderTimes = async (chefID: string) => {
    try {
      setIsLoading(true);
      const preOrderTime = await getChefPreOrderTimeService(chefID);
      const data = {
        days: Object.keys(preOrderTime).map((item) => ({
          label: convertEnglishWeekdayToPersian(item.toLocaleLowerCase()),
          value: item,
        })),
        times: {
          ...preOrderTime,
        },
      };
      return data;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  return {
    isLoading,
    getChefAvailableNow,
    getChefGallery,
    getPopupAlarms,
    getPaginatedChefs,
    getChefCenter,
    getChefAvailableTime,
    ChefRegistration,
    getChefPreOrderTimes,
  };
}

export default UseChefs;
