import {
  createTicketService,
  getTicketCategoriesService,
  showTicketService,
  userTicketService,
} from "@/src/services/httpServices/TicketServices";
import { FormOption, PaginationResult } from "@/src/types/general";
import { BaseTicket, TicketDetail, TicketItem } from "@/src/types/ticket";
import { convertApiTicketStatusToLocal } from "@/src/utils/Helpers";
import { useState } from "react";

function UseTicket() {
  const [isLoading, setIsLoading] = useState(false);

  const createTicket = async (data: BaseTicket, ticketId: string) => {
    try {
      setIsLoading(true);
      await createTicketService({
        category_id: data.categoryId,
        subject: data.title,
        text: data.description,
        file: data.file,
        ...(ticketId && {
          parent_id: ticketId,
        }),
      });
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const userTickets = async (page: number = 1) => {
    try {
      setIsLoading(true);
      const res = await userTicketService({
        page,
        paginate: 10,
      });
      const result: PaginationResult<TicketItem> = {
        currentPage: res.current_page,
        perPage: res.per_page,
        total: res.total,
        totalPages: res.last_page,
        data: res.data.map((item) => ({
          categoryId: item.category_id,
          description: item.text,
          categoryTitle: item?.category?.title ?? "",
          title: item.subject,
          id: item.id,
          status: convertApiTicketStatusToLocal(item.status),
        })),
      };
      return result;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const getTicketsCategories = async () => {
    try {
      setIsLoading(true);
      const result = await getTicketCategoriesService();
      const modeledData: FormOption[] = result.map((item) => ({
        value: item.id,
        label: item.title,
      }));
      return modeledData;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };
  const getTicketsDetail = async (id: string) => {
    try {
      setIsLoading(true);
      const result = await showTicketService(id);
      const modeledData: TicketDetail = {
        parent: {
          categoryId: result.category_id,
          description: result.text,
          categoryTitle: result?.category?.title ?? "",
          title: result.subject,
          id: result.id,
          status: convertApiTicketStatusToLocal(result.status),
          date: result.created_at,
          user: `${result.user?.name} ${result.user?.family}`,
          file: result.file,
        },
        child:
          result.child?.map((item) => ({
            categoryId: item.category_id,
            description: item.text,
            categoryTitle: item.category?.title ?? "",
            title: item.subject,
            id: item.id,
            status: convertApiTicketStatusToLocal(item.status),
            date: item.created_at,
            user: `${item.user?.name} ${item.user?.family}`,
            file: item.file,
          })) ?? [],
      };
      return modeledData;
    } catch (error) {
      console.log(error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  return {
    isLoading,
    createTicket,
    userTickets,
    getTicketsCategories,
    getTicketsDetail,
  };
}

export default UseTicket;
