import axios, { AxiosInstance, AxiosRequestConfig, AxiosResponse } from "axios";
import Cookies from "js-cookie";

interface ApiResponseData {
  data: {
    data: any;
  };
}

// Define a generic type for errors
export interface ApiDefaultError {
  message: string;
}

interface CustomAxiosConfig extends AxiosRequestConfig {
  hasToast?: boolean;
  isExternalApi?: boolean;
  endpoint?: string;
}

class HttpService<TError extends ApiDefaultError> {
  private axiosService: AxiosInstance;

  constructor() {
    this.axiosService = axios.create({
      baseURL: process.env.NEXT_PUBLIC_API_URL || "",
      headers: {
        "Content-Type": "application/json",
      },
    });

    this.axiosService.interceptors.request.use(async (request) => {
      const token = Cookies.get("token") || "";
      const guestToken = Cookies.get("Guest-token") || "";
      if (token) {
        request.headers.Authorization = `Bearer ${token}`;
      }
      if (guestToken) {
        request.headers.GuestToken = guestToken;
      }

      return request;
    });

    // Response interceptor
    this.axiosService.interceptors.response.use(
      (response: AxiosResponse<ApiResponseData>) => response,
      (error) => {
        console.error(`we got error in ${error.config.url}`);
        return Promise.reject((error.response.data as TError) || error.message);
      }
    );
  }

  async get<T = any>(
    url: string,
    config?: CustomAxiosConfig,
    parsData: boolean = true
  ): Promise<T> {
    const response = await this.axiosService.get(url, config);
    return parsData ? response?.data.data : response?.data;
  }

  async post<T = any>(
    url: string,
    data?: any,
    config?: CustomAxiosConfig
  ): Promise<T> {
    const response = await this.axiosService.post(url, data, config);
    return response?.data.data;
  }

  async put<T = any>(
    url: string,
    data?: any,
    config?: CustomAxiosConfig
  ): Promise<T> {
    const response = await this.axiosService.put(url, data, config);
    return response?.data.data;
  }

  async delete<T = any>(url: string, config?: CustomAxiosConfig): Promise<T> {
    const response = await this.axiosService.delete(url, config);
    return response?.data.data;
  }
}

// Create an instance of HttpService with the ApiDefaultError type
const httpService = new HttpService<ApiDefaultError>();

export { httpService };
