import { useEffect, useState } from "react";
import { useFormContext, Controller } from "react-hook-form";

interface Props {
    name: string;
    label?: string;
}

export default function ImageUpload({ name, label }: Props) {
    const { control } = useFormContext();
    const [preview, setPreview] = useState<string | null>(null);
    const [isDragging, setIsDragging] = useState(false);

    return (
        <div className="flex flex-col gap-4">
            {label && <label className="font-medium">{label}</label>}
            <Controller
                control={control}
                name={name}
                render={({ field }) => {
                    const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
                        const file = e.target.files?.[0];
                        if (file) {
                            field.onChange(e.target.files);
                            const reader = new FileReader();
                            reader.onloadend = () => setPreview(reader.result as string);
                            reader.readAsDataURL(file);
                        }
                    };

                    const handleDragOver = (e: React.DragEvent) => {
                        e.preventDefault();
                        setIsDragging(true);
                    };

                    const handleDragLeave = (e: React.DragEvent) => {
                        e.preventDefault();
                        setIsDragging(false);
                    };

                    const handleDrop = (e: React.DragEvent) => {
                        e.preventDefault();
                        setIsDragging(false);
                        const files = e.dataTransfer.files;

                        if (files.length > 0) {
                            const dt = new DataTransfer();
                            dt.items.add(files[0]);
                            const newFiles = dt.files;

                            field.onChange(newFiles);
                            const reader = new FileReader();
                            reader.onloadend = () => setPreview(reader.result as string);
                            reader.readAsDataURL(files[0]);
                        }
                    };

                    const handleRemove = () => {
                        setPreview(null);
                        field.onChange(null);
                        // Clear the file input
                        const input = document.getElementById(`fileInput-${name}`) as HTMLInputElement;
                        if (input) input.value = '';
                    };

                    return (
                        <div className="relative">
                            <label
                                htmlFor={`fileInput-${name}`}
                                className={`group flex relative border-2 border-dashed rounded-lg p-8 text-center cursor-pointer transition-colors ${isDragging && !preview
                                        ? 'border-blue-500 bg-blue-50'
                                        : 'border-gray-300 hover:border-gray-400'
                                    } ${preview ? 'h-60' : ''}`}
                                onDragOver={handleDragOver}
                                onDragLeave={handleDragLeave}
                                onDrop={handleDrop}
                            >
                                {preview ? (
                                    <div className="absolute inset-0 flex items-center justify-center">
                                        <img
                                            src={preview}
                                            alt="Preview"
                                            className="w-full h-full object-contain rounded-lg"
                                        />
                                        <button
                                            type="button"
                                            onClick={handleRemove}
                                            className="absolute top-2 right-2 p-1 bg-red-500 rounded-full hover:bg-red-600 transition-colors"
                                            aria-label="Remove image"
                                        >
                                            <svg
                                                className="w-6 h-6 text-black"
                                                fill="none"
                                                stroke="currentColor"
                                                viewBox="0 0 24 24"
                                                xmlns="http://www.w3.org/2000/svg"
                                            >
                                                <path
                                                    strokeLinecap="round"
                                                    strokeLinejoin="round"
                                                    strokeWidth={2}
                                                    d="M6 18L18 6M6 6l12 12"
                                                />
                                            </svg>
                                        </button>
                                    </div>
                                ) : (
                                    <div className="flex flex-col items-center gap-2">
                                        <svg
                                            className="w-12 h-12 text-gray-400 group-hover:text-gray-500"
                                            fill="none"
                                            stroke="currentColor"
                                            viewBox="0 0 24 24"
                                            xmlns="http://www.w3.org/2000/svg"
                                        >
                                            <path
                                                strokeLinecap="round"
                                                strokeLinejoin="round"
                                                strokeWidth={2}
                                                d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z"
                                            />
                                        </svg>
                                        <span className="text-gray-600 text-h5 group-hover:text-gray-700">
                                            برای بارگذاری تصویر کلیک کنید یا بکشید و رها کنید
                                        </span>
                                    </div>
                                )}
                            </label>
                            <input
                                id={`fileInput-${name}`}
                                type="file"
                                accept="image/*"
                                onChange={handleChange}
                                className="hidden"
                            />
                        </div>
                    );
                }}
            />
        </div>
    );
}