import React, { useRef, useEffect } from "react";
import { MapContainer, TileLayer, useMap } from "react-leaflet";
import L, { LatLngExpression } from "leaflet";
import Image from "next/image";
import pinImage from "@/public/assets/images/icons/marker.svg";
import "leaflet/dist/leaflet.css";

interface StaticMapProps {
  center: LatLngExpression;
  marker?: LatLngExpression;
}

function SetMapCenter({ center }: { center: LatLngExpression }) {
  const map = useMap();
  useEffect(() => {
    map.setView(center, map.getZoom());
  }, [center, map]);
  return null;
}

export default function StaticMap({ center }: StaticMapProps) {
  const pinIcon = useRef<L.Icon | null>(null);

  useEffect(() => {
    if (typeof window !== "undefined") {
      pinIcon.current = L.icon({
        iconUrl: pinImage.src,
        iconSize: [25, 41],
        iconAnchor: [12, 41],
      });
    }
  }, []);

  const openInMaps = () => {
    const [lat, lng] = center as [number, number];
    const scheme = /iPad|iPhone|iPod/.test(navigator.userAgent)
      ? `maps://maps.google.com/?q=${lat},${lng}`
      : `https://www.google.com/maps/dir/?api=1&destination=${lat},${lng}`;
    window.open(scheme, "_blank");
  };

  return (
    <div className="relative size-full">
      <MapContainer
        center={center}
        zoom={14}
        className="rounded-2xl size-full overflow-hidden"
        zoomControl={false}
        dragging={false}
        touchZoom={false}
        scrollWheelZoom={false}
        doubleClickZoom={false}
        attributionControl={false}
      >
        <TileLayer url="https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png" />
        <SetMapCenter center={center} />

        {/* Marker icon */}
        <div className="pointer-events-none absolute left-1/2 top-1/2 z-30 -translate-x-1/2 -translate-y-1/2">
          <Image
            src={pinImage}
            width={70}
            height={70}
            alt="marker"
            className="pointer-events-none -translate-x-1/2 -translate-y-full transform"
          />
        </div>
      </MapContainer>

      {/* 🧭 Navigate Button */}
      <button
        onClick={openInMaps}
        className="absolute bottom-4 right-4 z-[1000] flex items-center gap-2 rounded-3 bg-primary px-8 py-4 font-iranSans text-h5 text-white shadow-4 transition-all duration-200 hover:shadow-6 active:scale-[0.97]"
      >
        مسیریابی
      </button>
    </div>
  );
}
