import { useCartStore } from "@/src/store";
import { FoodMenu, Variety } from "@/src/types/food";
import React from "react";
import { useCart } from "@/src/hooks";
import { numberWithCommas } from "@/src/utils/Helpers";
import Link from "next/link";
import { useSnackbar } from "notistack";
import AddCardButton from "../AddCardButton";
import { ProductTumbSlider } from "../ProductTumbSlider";

interface Props {
  targetFood: FoodMenu;
  available?: boolean;
  onClose: VoidFunction;
}
function ProductPopupContent({ targetFood, available = true, onClose }: Props) {
  const { handleAddCard, handleRemoveCard, isLoading } = useCart();
  const { itemCount } = useCartStore();
  const { enqueueSnackbar } = useSnackbar();

  const AddCartHandler = async (product: FoodMenu, variety: Variety) => {
    try {
      await handleAddCard(product, variety);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };
  const removeFromCartHandler = async (varietyID: string) => {
    try {
      await handleRemoveCard(varietyID);
    } catch (error) {
      enqueueSnackbar(error?.message ?? "خطا در ثبت سفارش", {
        variant: "error",
      });
    }
  };
  return (
    <div className="flex  flex-col gap-8 lg:flex-row">
      <div className="w-full items-center justify-center lg:w-1/2">
        <ProductTumbSlider
          gallery={targetFood?.gallery ?? []}
          image={targetFood.image}
        />
      </div>
      <div className="flex flex-col gap-4  ">
        <Link
          onClick={() => onClose()}
          href={`/product/${targetFood.id}/${targetFood.title.replaceAll(" ", "-").replaceAll("%", "درصد")}`}
          className="text-h3 "
        >
          {targetFood?.title}
        </Link>
        <p className="max-h-80 overflow-y-auto text-h5 text-black/80">
          {targetFood?.Summary}
        </p>

        {available &&
          targetFood.varieties.map((item) => (
            <div
              key={item.varietyID}
              className="flex items-center justify-between"
            >
              <div>
                <span className="text-h5">{item.title}</span>
                <div className="flex items-center gap-2">
                  {item.discount?.percent !== 0 && (
                    <span className="text-h5  font-light text-secondary line-through ">
                      <span>{numberWithCommas(item.price)}</span>
                    </span>
                  )}
                  <p className="text-h3 font-bold">
                    {numberWithCommas(item.discount?.discountedPrice ?? 0)}{" "}
                    <span className="text-h5 font-normal">تومان</span>
                  </p>
                </div>
              </div>
              <AddCardButton
                isLoading={isLoading}
                key={item.varietyID}
                onAddCard={() => AddCartHandler(targetFood, item)}
                onRemoveCard={() => removeFromCartHandler(item.varietyID)}
                productInCart={itemCount(item.varietyID)}
                limit={item.maxOrderLimit}
              />
            </div>
          ))}
      </div>
    </div>
  );
}

export default ProductPopupContent;
