import React, {
  InputHTMLAttributes,
  useCallback,
  useMemo,
  useState,
} from "react";
import { Controller, useFormContext } from "react-hook-form";
import { ClassNameValue, twMerge } from "tailwind-merge";
import { useDebouncedCallback } from "use-debounce";
import { digitsFaToEn } from "@persian-tools/persian-tools";

type Props = InputHTMLAttributes<HTMLInputElement> & {
  name: string;
  validator?: (value: string) => Promise<{ isValid: boolean }>;
  validatorError?: string;
  disabled?: boolean;
  label?: string;
  className?: ClassNameValue;
  WrapperClassName?: ClassNameValue;
  fullWidth?: boolean;
  onlyNumber?: boolean;
};

interface ValidationStatus {
  error: string;
  hasError: boolean;
}

export default function RHFTextField({
  name,

  type,
  validator,
  validatorError = "Validation failed",
  disabled = false,
  label,
  fullWidth,
  className,
  WrapperClassName,
  onlyNumber,
  ...other
}: Props) {
  const { control } = useFormContext();

  const defaultApiValidationStatus: ValidationStatus = useMemo(
    () => ({
      error: validatorError,
      hasError: false,
    }),
    [validatorError]
  );

  const [isValidating, setIsValidating] = useState<boolean>(false);
  const [validationStatus, setValidationStatus] = useState<ValidationStatus>(
    defaultApiValidationStatus
  );

  const clearErrors = () =>
    setValidationStatus((prevState) => ({
      ...prevState,
      hasError: false,
    }));

  const triggerErrors = () =>
    setValidationStatus((prevState) => ({
      ...prevState,
      hasError: true,
    }));

  const handleValidator = useDebouncedCallback(
    useCallback(async (value: string) => {
      if (!validator || !value) {
        setIsValidating(false);
        clearErrors();
        return;
      }

      try {
        setIsValidating(true);
        clearErrors();

        const checkResults = await validator(value);
        if (!checkResults.isValid) {
          triggerErrors();
        } else {
          clearErrors();
        }
      } catch (error) {
        console.log(error);
      } finally {
        setIsValidating(false);
      }
      // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []),
    500
  );
  const inputClassnames = twMerge(
    `block resize-none placeholder:text-stroke-primary ${
      fullWidth ? "w-full" : "w-fit"
    }  py-4 px-8  text-h5 border 
    border-black/20 rounded-base bg-white placeholder-shown:text-stroke-primary     ${
      validationStatus.hasError ? "border-2 border-black/20 " : "border"
    }`,
    className
  );
  const allowOnlyNumber = (value: string) => value.replace(/[^0-9]/g, "");

  return (
    <Controller
      name={name}
      control={control}
      render={({ field, fieldState: { error, invalid } }) => (
        <div
          className={twMerge("flex max-w-md flex-col gap-3", WrapperClassName)}
        >
          {label && (
            <label className=" text-h4     " htmlFor={name}>
              {label}
            </label>
          )}

          <input
            {...field}
            type={type}
            id={name}
            autoComplete="off"
            value={type === "number" && field.value === 0 ? "" : field.value}
            onChange={(event) => {
              if (type === "number") {
                field.onChange(Number(event.target.value));
              } else {
                field.onChange(digitsFaToEn(event.target.value));
              }

              if (validator && !invalid) {
                handleValidator(event.target.value);
              }
              if (onlyNumber) {
                field.onChange(allowOnlyNumber(event.target.value));
              }
            }}
            disabled={disabled}
            {...other}
            className={twMerge(
              inputClassnames,
              !!error && "border-2 border-secondary"
            )}
          />
          {isValidating && <span>loading</span>}
          {!!error && (
            <p className=" mt-1 text-small text-primary">
              {error?.message ??
                (validationStatus.hasError && validationStatus.error)}
            </p>
          )}
        </div>
      )}
    />
  );
}
