import React, { FC, useEffect, useState } from "react";
import { Accordion } from "../Share/Accordion";
import { useAuthStore, useCartStore } from "@/src/store";
import { ServiceDetails } from "@/src/services/httpServices/CartServices/types";
import { DeliveryMethod } from "@/src/hooks/Queries/useDeliveryMethodByTime";
import { convertEnglishWeekdayToPersian } from "@/src/utils/Helpers";
import { useSelectionStore } from "@/src/store/useSelectionStore";
import { useRestaurantSelection } from "@/src/containers/useRestaurantSelection";

type props = {
  areaAddressData: ServiceDetails | undefined;
  chefId: string;
  deliveryType: DeliveryMethod | undefined;
  setAvailableModal: React.Dispatch<React.SetStateAction<boolean>>;
};
const RestaurantAccordion: FC<props> = ({
  areaAddressData,
  chefId,
  deliveryType,
  setAvailableModal,
}) => {
  console.log("🚀 ~ RestaurantAccordion ~ areaAddressData:", areaAddressData);
  const { deliveryTime } = useRestaurantSelection(chefId);
  const { ensureRestaurant, getSelection } = useSelectionStore();

  const translateDayToPersian = (day: string): string => {
    const days: Record<string, string> = {
      Saturday: "شنبه",
      Sunday: "یک‌شنبه",
      Monday: "دوشنبه",
      Tuesday: "سه‌شنبه",
      Wednesday: "چهارشنبه",
      Thursday: "پنج‌شنبه",
      Friday: "جمعه",
    };

    return days[day] || day;
  };
  const { authUser } = useAuthStore();
  const { addressOrder } = useCartStore();

  useEffect(() => {
    if (chefId) ensureRestaurant(chefId);
  }, [chefId, ensureRestaurant]);

  const selectedFromStore = getSelection(chefId)?.selectedAddressId;
  const selectedAddressId = selectedFromStore ?? authUser.defaultAddress;

  const selectedAddress = addressOrder?.find(
    (item) => String(item.value) === String(selectedAddressId)
  );

  return (
    <div>
      <Accordion defaultValue={["item-1"]}>
        <Accordion.Item value="item-1" label="انتخاب زمان دریافت سفارش">
          <div className="mb-4 mt-4 flex items-center text-h5">
            <div className="space-y-1">
              <p className="">
                <span className="">روش ارسال:</span>{" "}
                {deliveryType?.title
                  ? deliveryType?.title
                  : areaAddressData?.delivery_type?.title || "-----"}
              </p>
            </div>
            <span className="flex font-medium">
              <div className="ms-7 flex justify-center"></div>
            </span>
          </div>

          <div className="flex items-center py-4 text-h5">
            <span className="ml-3 whitespace-nowrap leading-[26px]">
              زمان ارسال :
            </span>

            {deliveryTime ? (
              deliveryTime.currentTime ? (
                // ✅ Case 1: currentTime = true → show "الان"
                <p className="text-base text-green-600 ml-2 font-semibold">
                  الان
                </p>
              ) : deliveryTime.time ? (
                // ✅ Case 2: normal scheduled delivery → show day + time
                <div className="ml-2 space-y-1">
                  <div className="flex text-h4 font-semibold text-gray-700">
                    <p className="mx-1">
                      {convertEnglishWeekdayToPersian(deliveryTime.day)}
                    </p>
                    <span className="mx-2">ساعت</span>
                    {deliveryTime.time}
                  </div>
                </div>
              ) : (
                // ✅ Case 3: deliveryTime exists but no time (edge case)
                <p className="text-base text-red-500 ml-2 font-semibold">
                  الان
                </p>
              )
            ) : (
              // ✅ Case 4: no deliveryTime at all (restaurant closed fallback)
              <div className="ml-2">
                <span className="mt-2 flex items-center font-medium">
                  {translateDayToPersian(areaAddressData?.data?.day as string)}{" "}
                  <span
                    className={`mx-2 ${
                      areaAddressData?.data?.service_time_from ? "" : "hidden"
                    }`}
                  >
                    ساعت
                  </span>
                  {areaAddressData?.data?.service_time_from?.slice(0, 5)}
                </span>
                <span className="text-[10px] text-gray-500">
                  رستوران در حال حاضر بسته است و نزدیک‌ترین زمان برای شما انتخاب
                  شده است
                </span>
              </div>
            )}
          </div>

          <div className="flex items-center py-4 text-h5">
            {authUser && Object.values(authUser).some((value) => value) && (
              <span className="ml-3 whitespace-nowrap leading-[26px]">
                آدرس :{" "}
              </span>
            )}

            <span className="mx-9 flex flex-col items-center ">
              <p>
                {selectedAddress?.description ??
                  authUser.defaultAddressItem?.address ??
                  ""}
              </p>
              {areaAddressData?.address_valid ? (
                ""
              ) : (
                <p className=" text-darkBlue">
                  {areaAddressData?.message || ""}
                </p>
              )}
            </span>
            <div className="ms-7 flex justify-center"></div>
          </div>
          <button
            onClick={() => setAvailableModal(true)}
            className="text-primary-main mb-4 w-full rounded-half border border-primary px-4 py-3 text-h5 transition-colors duration-200 hover:bg-primary hover:text-white"
          >
            تغییر شیوه دریافت
          </button>
        </Accordion.Item>
      </Accordion>
    </div>
  );
};

export default RestaurantAccordion;
