import { ArrowDown2 } from 'iconsax-react';
import React, {
  createContext,
  useContext,
  useState,
  ReactNode,
  FC,
} from 'react';

type AccordionContextType = {
  opened: string[];
  toggle: (value: string) => void;
};

const AccordionContext = createContext<AccordionContextType | undefined>(undefined);

// -------------------- Accordion Root --------------------
type AccordionRootProps = {
  children: ReactNode;
  multiple?: boolean;
  defaultValue?: string[];
};

const AccordionRoot: FC<AccordionRootProps> = ({
  children,
  multiple = false,
  defaultValue = [],
}) => {
  const [opened, setOpened] = useState<string[]>(defaultValue);

  const toggle = (value: string) => {
    setOpened((prev) =>
      prev.includes(value)
        ? prev.filter((item) => item !== value)
        : multiple
        ? [...prev, value]
        : [value]
    );
  };

  return (
    <AccordionContext.Provider value={{ opened, toggle }}>
      <div className="rounded-half border divide-y">{children}</div>
    </AccordionContext.Provider>
  );
};

// -------------------- Accordion Item --------------------
type AccordionItemProps = {
  value: string;
  label: ReactNode;
  children: ReactNode;
};

const AccordionItem: FC<AccordionItemProps> = ({ value, label, children }) => {
  const context = useContext(AccordionContext);
  if (!context) {
    throw new Error('Accordion.Item must be used within Accordion');
  }

  const { opened, toggle } = context;
  const isOpen = opened.includes(value);

  return (
    <div>
      <button
        className="w-full flex items-center justify-between px-4 py-6 text-left font-medium hover:bg-gray-50 transition"
        onClick={() => toggle(value)}
      >
        <span className='text-h4'>{label}</span>
        <ArrowDown2
          className={`h-5 w-5 transition-transform duration-300 ${
            isOpen ? 'rotate-180' : ''
          }`}
        />
      </button>

      <div
        className={`overflow-hidden transition-all duration-300 px-4 ${
          isOpen ? 'max-h-96 py-2' : 'max-h-0'
        }`}
      >
        <div className="text-gray-700 text-sm">{children}</div>
      </div>
    </div>
  );
};

// -------------------- Export Compound Component --------------------
export const Accordion = Object.assign(AccordionRoot, {
  Item: AccordionItem,
});
