import React from "react";
import { useForm, FormProvider } from "react-hook-form";
import RHFSelect from "../RHF/Select";
export interface DayTimePickerDefaultValues {
  day: string;
  hour: number;
  minute?: number;
}

export default function DayTimePickerForm({
  defaultValues,
  loadingShow,
  setShowPicker,
  onSubmit,
}: {
  defaultValues?: DayTimePickerDefaultValues;
  onSubmit: (data: DayTimePickerDefaultValues) => void;
  loadingShow: boolean;
  setShowPicker: React.Dispatch<React.SetStateAction<boolean>>;
}) {
  const dayOptions = [
    { label: "شنبه", value: "saturday" },
    { label: "یکشنبه", value: "sunday" },
    { label: "دوشنبه", value: "monday" },
    { label: "سه‌شنبه", value: "tuesday" },
    { label: "چهارشنبه", value: "wednesday" },
    { label: "پنجشنبه", value: "thursday" },
    { label: "جمعه", value: "friday" },
  ];

  const hourOptions = Array.from({ length: 24 }, (_, i) => ({
    label: i.toString().padStart(2, "0"),
    value: i,
  }));

  const minuteOptions = Array.from({ length: 60 }, (_, i) => ({
    label: i.toString().padStart(2, "0"),
    value: i,
  }));
  const defaultFormValues = {
    day: dayOptions.find((opt) => opt.value === defaultValues?.day),
    hour: hourOptions.find((opt) => opt.value === defaultValues?.hour),
    minute: minuteOptions.find((opt) => opt.value === defaultValues?.minute),
  };
  const methods = useForm({ defaultValues: defaultFormValues });
  const { handleSubmit } = methods;

  return (
    <FormProvider {...methods}>
      <form
        onSubmit={handleSubmit((formValues) => {
          onSubmit({
            day: formValues.day?.value ?? "",
            hour: formValues.hour?.value ?? 0,
            minute: formValues.minute?.value,
          });
          setShowPicker(false);
        })}
        className="rounded-2xl shadow-md max-w-md space-y-4 bg-white"
      >

        <div className="grid grid-cols-3 gap-4 sm:grid-cols-3">
          <RHFSelect name="day" label="روز" options={dayOptions} fullWidth />
          <RHFSelect name="hour" label="ساعت" options={hourOptions} fullWidth />
          <RHFSelect
            name="minute"
            label="دقیقه"
            options={minuteOptions}
            fullWidth
          />
        </div>
        <div className="flex justify-end">
          <button
            type="submit"
            disabled={loadingShow}
            className="rounded hover:bg-primary-dark mt-4 flex h-[33px] w-[87px] items-center justify-center gap-2 bg-primary px-4 py-2 text-white"
          >
            {loadingShow ? (
              <span className=" flex items-center justify-start ">
                <svg
                  className="h-5 w-5 animate-spin text-white"
                  xmlns="http://www.w3.org/2000/svg"
                  fill="none"
                  viewBox="0 0 24 24"
                >
                  <circle
                    className="opacity-25"
                    cx="12"
                    cy="12"
                    r="10"
                    stroke="currentColor"
                    strokeWidth="4"
                  ></circle>
                  <path
                    className="opacity-75"
                    fill="currentColor"
                    d="M4 12a8 8 0 018-8v4a4 4 0 00-4 4H4z"
                  ></path>
                </svg>
              </span>
            ) : (
              <span>تایید زمان</span>
            )}
          </button>
        </div>
      </form>
    </FormProvider>
  );
}
